// GHL Utils - API Wrapper
// Routes API calls to either Revex or PIT based on user settings

// Check auth method and route to appropriate API
async function getAuthMethod() {
  return new Promise((resolve) => {
    chrome.storage.sync.get(['authMethod'], (result) => {
      resolve(result.authMethod || 'revex'); // Default to revex
    });
  });
}

// GET request wrapper
async function apiGet(endpoint) {
  const authMethod = await getAuthMethod();

  if (authMethod === 'pit') {
    return pitGet(endpoint);
  } else {
    return revexGet(endpoint);
  }
}

// POST request wrapper
async function apiPost(endpoint, data) {
  const authMethod = await getAuthMethod();

  if (authMethod === 'pit') {
    return pitPost(endpoint, data);
  } else {
    return revexPost(endpoint, data);
  }
}

// PUT request wrapper
async function apiPut(endpoint, data) {
  const authMethod = await getAuthMethod();

  if (authMethod === 'pit') {
    return pitPut(endpoint, data);
  } else {
    return revexPut(endpoint, data);
  }
}

// PIT API functions
async function pitGet(endpoint) {
  const { ghlApiKey } = await chrome.storage.sync.get(['ghlApiKey']);

  if (!ghlApiKey) {
    throw new Error('Private Integration Token not configured');
  }

  const response = await fetch(`https://services.leadconnectorhq.com${endpoint}`, {
    method: 'GET',
    headers: {
      'Authorization': `Bearer ${ghlApiKey}`,
      'Version': '2021-07-28',
      'Content-Type': 'application/json'
    }
  });

  if (!response.ok) {
    throw new Error(`API error: ${response.status}`);
  }

  const data = await response.json();
  return { data, status: response.status };
}

async function pitPost(endpoint, postData) {
  const { ghlApiKey } = await chrome.storage.sync.get(['ghlApiKey']);

  if (!ghlApiKey) {
    throw new Error('Private Integration Token not configured');
  }

  const response = await fetch(`https://services.leadconnectorhq.com${endpoint}`, {
    method: 'POST',
    headers: {
      'Authorization': `Bearer ${ghlApiKey}`,
      'Version': '2021-07-28',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify(postData)
  });

  if (!response.ok) {
    throw new Error(`API error: ${response.status}`);
  }

  const data = await response.json();
  return { data, status: response.status };
}

async function pitPut(endpoint, putData) {
  const { ghlApiKey } = await chrome.storage.sync.get(['ghlApiKey']);

  if (!ghlApiKey) {
    throw new Error('Private Integration Token not configured');
  }

  const response = await fetch(`https://services.leadconnectorhq.com${endpoint}`, {
    method: 'PUT',
    headers: {
      'Authorization': `Bearer ${ghlApiKey}`,
      'Version': '2021-07-28',
      'Content-Type': 'application/json'
    },
    body: JSON.stringify(putData)
  });

  if (!response.ok) {
    throw new Error(`API error: ${response.status}`);
  }

  const data = await response.json();
  return { data, status: response.status };
}

// Revex API functions (delegate to revex-auth.js)
async function revexGet(endpoint) {
  if (!window.ghlUtilsRevex) {
    throw new Error('Revex not initialized');
  }
  return window.ghlUtilsRevex.get(endpoint);
}

async function revexPost(endpoint, data) {
  if (!window.ghlUtilsRevex) {
    throw new Error('Revex not initialized');
  }
  return window.ghlUtilsRevex.post(endpoint, data);
}

async function revexPut(endpoint, data) {
  if (!window.ghlUtilsRevex) {
    throw new Error('Revex not initialized');
  }
  return window.ghlUtilsRevex.put(endpoint, data);
}

// Get location ID
function getLocationId() {
  if (window.ghlUtilsRevex && window.ghlUtilsRevex.getLocationId) {
    return window.ghlUtilsRevex.getLocationId();
  }

  // Fallback: extract from URL
  const urlMatch = window.location.href.match(/\/location\/([A-Za-z0-9_-]{18,28})/);
  if (urlMatch && urlMatch[1]) {
    return urlMatch[1];
  }

  return null;
}

// Wait for Revex to be ready (only needed for Revex mode)
async function waitForReady() {
  const authMethod = await getAuthMethod();

  if (authMethod === 'revex') {
    if (window.ghlUtilsRevex && window.ghlUtilsRevex.waitForReady) {
      return window.ghlUtilsRevex.waitForReady();
    }
  }

  // PIT doesn't need to wait
  return Promise.resolve();
}

// Export unified API
window.ghlUtilsAPI = {
  get: apiGet,
  post: apiPost,
  put: apiPut,
  getLocationId: getLocationId,
  waitForReady: waitForReady
};

console.log('[API Wrapper] Initialized');
