// Background service worker for GHL Utils

// GHL URL patterns
const GHL_PATTERNS = {
  workflow: 'https://app.gohighlevel.com/location/{locationId}/workflow/{id}',
  page: 'https://app.gohighlevel.com/v2/location/{locationId}/funnels-websites/funnels/{id}',
  form: 'https://app.gohighlevel.com/v2/location/{locationId}/form-builder-v2/{id}',
  survey: 'https://app.gohighlevel.com/v2/location/{locationId}/survey-builder-v2/{id}',
  contact: 'https://app.gohighlevel.com/v2/location/{locationId}/contacts/detail/{id}',
  calendar: 'https://app.gohighlevel.com/v2/location/{locationId}/calendars/{id}', // Placeholder - needs actual URL
  integration: 'https://app.gohighlevel.com/v2/location/{locationId}/settings/integrations/{id}', // Placeholder - needs actual URL
  audit: 'https://app.gohighlevel.com/v2/location/{locationId}/settings/audit/logs?documentId={id}'
};

// Initialize context menus
chrome.runtime.onInstalled.addListener(() => {
  createContextMenus();
});

function createContextMenus() {
  chrome.contextMenus.removeAll(() => {
    chrome.contextMenus.create({
      id: 'ghl-open-workflow',
      title: 'Open as Workflow: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-open-page',
      title: 'Open as Page/Funnel: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-open-form',
      title: 'Open as Form: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-open-survey',
      title: 'Open as Survey: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-open-contact',
      title: 'Open as Contact: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-open-audit',
      title: 'Open Audit Logs: "%s"',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-separator',
      type: 'separator',
      contexts: ['selection']
    });

    chrome.contextMenus.create({
      id: 'ghl-copy-id',
      title: 'Copy ID: "%s"',
      contexts: ['selection']
    });
  });
}

// Handle context menu clicks
chrome.contextMenus.onClicked.addListener((info, tab) => {
  const selectedText = info.selectionText.trim();

  if (info.menuItemId === 'ghl-copy-id') {
    // Copy is handled by the browser automatically
    return;
  }

  // Map menu item to resource type
  const typeMap = {
    'ghl-open-workflow': 'workflow',
    'ghl-open-page': 'page',
    'ghl-open-form': 'form',
    'ghl-open-survey': 'survey',
    'ghl-open-contact': 'contact',
    'ghl-open-audit': 'audit'
  };

  const type = typeMap[info.menuItemId];
  if (type) {
    handleNavigation(type, selectedText);
  }
});

// Listen for command (keyboard shortcut)
chrome.commands.onCommand.addListener((command) => {
  if (command === 'open-command-palette') {
    // Send message to active tab to open command palette
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, { action: 'toggleCommandPalette' });
      }
    });
  } else if (command === 'open-page-builder') {
    // Send message to active tab to open page builder
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, { action: 'openPageBuilder' });
      }
    });
  }
});

// Listen for navigation requests from content script
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'navigate') {
    handleNavigation(request.type, request.id, request.contactId);
  } else if (request.action === 'openGuide') {
    // Open PIT setup guide in new tab
    chrome.tabs.create({
      url: chrome.runtime.getURL('pit-guide.html')
    });
    sendResponse({ success: true });
  } else if (request.action === 'openSettings' || request.action === 'openPopup') {
    // Try to open the popup (browser limitation - may not work)
    // Instead, show a notification to click the extension icon
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.sendMessage(tabs[0].id, {
          action: 'showToast',
          message: '👆 Click the GHL Utils icon in your toolbar to open settings'
        });
      }
    });
    sendResponse({ success: true });
  } else if (request.action === 'openTabNextToCurrent') {
    // Open tab next to current tab
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        chrome.tabs.create({
          url: request.url,
          index: tabs[0].index + 1,
          active: true
        });
      }
    });
  } else if (request.action === 'updateBadge') {
    // Update badge when GHL page is detected
    const tabId = sender.tab?.id;

    if (!tabId) {
      return;
    }

    if (request.isGHLPage) {
      // Set a green badge when on a GHL page
      chrome.action.setBadgeText({
        text: '✓',
        tabId: tabId
      });

      chrome.action.setBadgeBackgroundColor({
        color: '#4CAF50',
        tabId: tabId
      });

      chrome.action.setTitle({
        title: 'GHL Utils - GHL Page Detected!',
        tabId: tabId
      });
    } else {
      // Clear badge when not on a GHL page
      chrome.action.setBadgeText({
        text: '',
        tabId: tabId
      });

      chrome.action.setTitle({
        title: 'GHL Utils',
        tabId: tabId
      });
    }

    sendResponse({ success: true });
  }
});

async function handleNavigation(type, id, contactId = null) {
  // Get location ID from storage
  const result = await chrome.storage.sync.get(['locationId']);
  const locationId = result.locationId;

  if (!locationId) {
    console.error('Location ID not set');
    return;
  }

  let url = buildUrl(type, id, locationId);

  // Append contactId parameter if provided
  if (url && contactId) {
    // For workflows, add /status before the contactId parameter
    if (type === 'workflow') {
      url += `/status?contactId=${contactId}`;
    } else {
      url += `?contactId=${contactId}`;
    }
  }

  if (url) {
    chrome.tabs.create({ url });
  }
}

function buildUrl(type, id, locationId) {
  const pattern = GHL_PATTERNS[type];
  if (!pattern) return null;

  return pattern
    .replace('{locationId}', locationId)
    .replace('{id}', id);
}


// Clear badge when tab is updated (new page)
chrome.tabs.onUpdated.addListener((tabId, changeInfo, tab) => {
  if (changeInfo.status === 'loading') {
    chrome.action.setBadgeText({
      text: '',
      tabId: tabId
    });
  }
});
