// GHL Utils - Bulk Contact Field Editor
// Allows users to bulk edit contact fields without Excel export/import

// State management
let bulkEditorState = {
  selectedContacts: [],
  locationId: null,
  customFields: [],
  isProcessing: false
};

// Initialize bulk contact editor
async function initBulkContactEditor() {
  // Get stored location ID
  const stored = await chrome.storage.sync.get(['locationId']);
  bulkEditorState.locationId = stored.locationId;

  // Only initialize if on contacts page
  if (!window.location.href.includes('/contacts')) {
    return;
  }

  // Monitor for contact selection changes
  startContactSelectionMonitor();
}

// Monitor for selected contacts
function startContactSelectionMonitor() {
  let lastSelectedCount = 0;

  const checkSelection = () => {
    // Find all checked checkboxes, but ONLY those that are in actual contact rows
    const allCheckedBoxes = Array.from(document.querySelectorAll('input[type="checkbox"]:checked:not([disabled])'));

    // Filter to only checkboxes that are in actual table rows or contact cards
    const selectedCheckboxes = allCheckedBoxes.filter(cb => {
      // Must have a parent row
      const hasParentRow = cb.closest('tr, [role="row"], [class*="row"]') !== null;

      // Exclude checkboxes in headers
      const isInHeader = cb.closest('thead, [role="columnheader"]') !== null;

      // Exclude filter/search checkboxes (typically have specific IDs like "name", "email", etc.)
      const hasFilterId = ['name', 'email', 'phone', 'tags', 'status'].includes(cb.id);

      // Exclude "select all" checkboxes
      const isSelectAll =
        cb.getAttribute('aria-label')?.toLowerCase().includes('select all') ||
        cb.id?.toLowerCase().includes('select-all') ||
        cb.id?.toLowerCase().includes('selectall');

      return hasParentRow && !isInHeader && !hasFilterId && !isSelectAll;
    });

    // Extract contact IDs from selected rows
    const selectedContacts = [];
    selectedCheckboxes.forEach((checkbox) => {
      // Try to find contact ID from various sources
      let contactId =
        checkbox.getAttribute('data-contact-id') ||
        checkbox.getAttribute('data-id') ||
        checkbox.getAttribute('value') ||
        checkbox.getAttribute('id') ||
        findContactIdFromRow(checkbox);

      // Clean up the ID (remove any prefixes like "contact-")
      if (contactId) {
        contactId = contactId.replace(/^(contact-|id-|row-)/, '');
      }

      if (contactId && contactId.length >= 16 && contactId.length <= 32) {
        selectedContacts.push(contactId);
      }
    });

    // Remove duplicates
    bulkEditorState.selectedContacts = [...new Set(selectedContacts)];

    // Show/hide bulk edit button based on selection
    if (bulkEditorState.selectedContacts.length > 0 && bulkEditorState.selectedContacts.length !== lastSelectedCount) {
      showBulkEditButton();
      lastSelectedCount = bulkEditorState.selectedContacts.length;
    } else if (bulkEditorState.selectedContacts.length === 0 && lastSelectedCount > 0) {
      hideBulkEditButton();
      lastSelectedCount = 0;
    }
  };

  // Check immediately with multiple delays to catch different load states
  setTimeout(checkSelection, 1000);
  setTimeout(checkSelection, 3000);
  setTimeout(checkSelection, 5000);

  // Monitor for changes using MutationObserver
  const observer = new MutationObserver(() => {
    checkSelection();
  });

  // Observe the contacts table area
  observer.observe(document.body, {
    childList: true,
    subtree: true,
    attributes: true,
    attributeFilter: ['checked', 'aria-checked']
  });

  // Also check on click events
  document.addEventListener('click', (e) => {
    if (e.target.type === 'checkbox') {
      setTimeout(checkSelection, 100);
      setTimeout(checkSelection, 500); // Check again after a delay
    }
  }, true);

  // Also check on change events
  document.addEventListener('change', (e) => {
    if (e.target.type === 'checkbox') {
      setTimeout(checkSelection, 100);
    }
  }, true);
}

// Find contact ID from checkbox's parent row
function findContactIdFromRow(checkbox) {
  // Try multiple row detection strategies
  let row = checkbox.closest('tr') ||
            checkbox.closest('[role="row"]') ||
            checkbox.closest('[class*="row"]') ||
            checkbox.closest('[class*="Row"]') ||
            checkbox.closest('[class*="item"]') ||
            checkbox.closest('[class*="Item"]');

  if (!row) {
    return null;
  }

  // Strategy 1: Check row's data attributes
  const dataAttributes = ['data-contact-id', 'data-id', 'data-row-id', 'data-key', 'data-record-id'];
  for (const attr of dataAttributes) {
    const value = row.getAttribute(attr);
    if (value && value.length >= 16 && value.length <= 32) {
      return value;
    }
  }

  // Strategy 2: Look for ID in row's child elements (links, buttons, etc.)
  const links = row.querySelectorAll('a[href*="/contacts/detail/"]');
  if (links.length > 0) {
    const match = links[0].href.match(/\/contacts\/detail\/([A-Za-z0-9_-]{16,32})/);
    if (match) {
      return match[1];
    }
  }

  // Strategy 3: Check for any child elements with data-id attributes
  const elementsWithId = row.querySelectorAll('[data-id], [data-contact-id], [data-key]');
  for (const elem of elementsWithId) {
    const id = elem.getAttribute('data-id') ||
               elem.getAttribute('data-contact-id') ||
               elem.getAttribute('data-key');
    if (id && id.length >= 16 && id.length <= 32) {
      return id;
    }
  }

  // Strategy 4: Look for GHL ID pattern in row's text content
  const rowText = row.textContent || row.innerText;
  const contactIds = rowText.match(/\b[A-Za-z0-9]{20,32}\b/g);
  if (contactIds && contactIds.length > 0) {
    // Filter to only valid-looking IDs (mix of letters and numbers)
    const validId = contactIds.find(id =>
      /[a-zA-Z]/.test(id) && /[0-9]/.test(id) && id.length >= 20
    );
    if (validId) {
      return validId;
    }
  }

  // Strategy 5: Check all attributes on the row element itself
  if (row.attributes) {
    for (const attr of row.attributes) {
      const value = attr.value;
      if (value && value.length >= 16 && value.length <= 32 && /^[A-Za-z0-9_-]+$/.test(value)) {
        // Check if it looks like a GHL ID (has both letters and numbers)
        if (/[a-zA-Z]/.test(value) && /[0-9]/.test(value)) {
          return value;
        }
      }
    }
  }

  return null;
}

// Show bulk edit button
function showBulkEditButton() {
  // Remove existing button if any
  hideBulkEditButton();

  const button = document.createElement('button');
  button.id = 'ghl-utils-bulk-edit-btn';
  button.innerHTML = `
    <svg width="16" height="16" viewBox="0 0 16 16" fill="currentColor" style="margin-right: 6px;">
      <path d="M12.854 1.146a.5.5 0 0 0-.707 0L10.5 2.793 13.207 5.5l1.647-1.646a.5.5 0 0 0 0-.708l-2-2zm.646 3.854l-2.707-2.707L3.5 9.586V12h2.414l7.293-7.293z"/>
      <path d="M1 13.5A1.5 1.5 0 0 0 2.5 15h11a1.5 1.5 0 0 0 1.5-1.5v-6a.5.5 0 0 0-1 0v6a.5.5 0 0 1-.5.5h-11a.5.5 0 0 1-.5-.5v-11a.5.5 0 0 1 .5-.5H9a.5.5 0 0 0 0-1H2.5A1.5 1.5 0 0 0 1 2.5v11z"/>
    </svg>
    <span style="display: flex; align-items: center; gap: 6px;">
      Bulk Edit (${bulkEditorState.selectedContacts.length})
      <span style="
        background: rgba(255, 255, 255, 0.25);
        padding: 2px 6px;
        border-radius: 4px;
        font-size: 10px;
        font-weight: 700;
        letter-spacing: 0.5px;
      ">BETA</span>
    </span>
  `;
  button.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 80px;
    background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
    color: white;
    padding: 14px 24px;
    border: none;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 600;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    cursor: pointer;
    z-index: 99999998;
    box-shadow: 0 4px 16px rgba(251, 133, 0, 0.4);
    display: flex;
    align-items: center;
    transition: all 0.3s ease;
    animation: slideInUp 0.3s ease-out;
  `;

  button.addEventListener('mouseenter', () => {
    button.style.transform = 'translateY(-2px)';
    button.style.boxShadow = '0 6px 20px rgba(251, 133, 0, 0.5)';
  });

  button.addEventListener('mouseleave', () => {
    button.style.transform = 'translateY(0)';
    button.style.boxShadow = '0 4px 16px rgba(251, 133, 0, 0.4)';
  });

  button.addEventListener('click', () => {
    openBulkEditModal();
  });

  document.body.appendChild(button);
}

// Hide bulk edit button
function hideBulkEditButton() {
  const button = document.getElementById('ghl-utils-bulk-edit-btn');
  if (button) {
    button.remove();
  }
}

// Open bulk edit modal
async function openBulkEditModal() {
  // Fetch custom fields if not already loaded
  if (bulkEditorState.customFields.length === 0) {
    showToast('Loading custom fields...', 2000);
    await fetchCustomFields();
  }

  // Create modal overlay
  const overlay = document.createElement('div');
  overlay.id = 'ghl-utils-bulk-edit-modal';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.6);
    backdrop-filter: blur(4px);
    z-index: 999999999;
    display: flex;
    align-items: center;
    justify-content: center;
    animation: fadeIn 0.2s ease-out;
  `;

  overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
      closeBulkEditModal();
    }
  });

  // Create modal
  const modal = document.createElement('div');
  modal.style.cssText = `
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    max-height: 80vh;
    overflow-y: auto;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideInDown 0.3s ease-out;
  `;

  modal.innerHTML = `
    <div style="padding: 24px; border-bottom: 1px solid #e5e7eb;">
      <div style="display: flex; justify-content: space-between; align-items: center;">
        <h2 style="margin: 0; font-size: 20px; font-weight: 600; color: #111827; display: flex; align-items: center; gap: 10px;">
          <span>Bulk Edit ${bulkEditorState.selectedContacts.length} Contacts</span>
          <span style="
            background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
            color: white;
            padding: 3px 8px;
            border-radius: 4px;
            font-size: 11px;
            font-weight: 700;
            letter-spacing: 0.5px;
          ">BETA</span>
        </h2>
        <button id="ghl-utils-close-modal" style="
          background: none;
          border: none;
          font-size: 24px;
          color: #6b7280;
          cursor: pointer;
          padding: 0;
          width: 32px;
          height: 32px;
          display: flex;
          align-items: center;
          justify-content: center;
          border-radius: 4px;
          transition: all 0.2s;
        ">&times;</button>
      </div>
    </div>

    <div style="padding: 24px;">
      <div style="margin-bottom: 20px;">
        <label style="display: block; margin-bottom: 8px; font-weight: 500; color: #374151;">
          Select Field to Edit
        </label>
        <select id="ghl-utils-field-select" style="
          width: 100%;
          padding: 10px 12px;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          font-size: 14px;
          color: #111827;
          background: white;
          cursor: pointer;
        ">
          <option value="">Choose a field...</option>
          <optgroup label="Standard Fields">
            <option value="firstName">First Name</option>
            <option value="lastName">Last Name</option>
            <option value="email">Email</option>
            <option value="phone">Phone</option>
            <option value="address1">Address Line 1</option>
            <option value="city">City</option>
            <option value="state">State</option>
            <option value="postalCode">Postal Code</option>
            <option value="country">Country</option>
            <option value="companyName">Company Name</option>
            <option value="website">Website</option>
            <option value="tags">Tags (comma-separated)</option>
          </optgroup>
          ${bulkEditorState.customFields.length > 0 ? `
            <optgroup label="Custom Fields">
              ${bulkEditorState.customFields.map(field =>
                `<option value="customField.${field.id}">${field.name}</option>`
              ).join('')}
            </optgroup>
          ` : ''}
        </select>
      </div>

      <div style="margin-bottom: 20px;">
        <label style="display: block; margin-bottom: 8px; font-weight: 500; color: #374151;">
          Update Action
        </label>
        <select id="ghl-utils-action-select" style="
          width: 100%;
          padding: 10px 12px;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          font-size: 14px;
          color: #111827;
          background: white;
          cursor: pointer;
        ">
          <option value="set">Set to value</option>
          <option value="append">Append to existing</option>
          <option value="clear">Clear field</option>
        </select>
      </div>

      <div id="ghl-utils-value-container" style="margin-bottom: 20px;">
        <label style="display: block; margin-bottom: 8px; font-weight: 500; color: #374151;">
          New Value
        </label>
        <input type="text" id="ghl-utils-new-value" placeholder="Enter new value..." style="
          width: 100%;
          padding: 10px 12px;
          border: 1px solid #d1d5db;
          border-radius: 6px;
          font-size: 14px;
          color: #111827;
        "/>
        <p style="margin: 8px 0 0; font-size: 12px; color: #6b7280;">
          For tags, separate multiple tags with commas
        </p>
      </div>

      <div style="
        background: #f9fafb;
        border: 1px solid #e5e7eb;
        border-radius: 6px;
        padding: 12px;
        margin-bottom: 20px;
      ">
        <p style="margin: 0; font-size: 13px; color: #6b7280;">
          <strong style="color: #111827;">Selected:</strong> ${bulkEditorState.selectedContacts.length} contacts will be updated
        </p>
      </div>

      <div style="display: flex; gap: 12px; justify-content: flex-end;">
        <button id="ghl-utils-cancel-btn" style="
          padding: 10px 20px;
          border: 1px solid #d1d5db;
          background: white;
          color: #374151;
          border-radius: 6px;
          font-size: 14px;
          font-weight: 500;
          cursor: pointer;
          transition: all 0.2s;
        ">Cancel</button>
        <button id="ghl-utils-apply-btn" style="
          padding: 10px 20px;
          border: none;
          background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
          color: white;
          border-radius: 6px;
          font-size: 14px;
          font-weight: 500;
          cursor: pointer;
          transition: all 0.2s;
        ">Apply Changes</button>
      </div>

      <div id="ghl-utils-progress-container" style="display: none; margin-top: 20px;">
        <div style="
          background: #f3f4f6;
          border-radius: 8px;
          height: 8px;
          overflow: hidden;
        ">
          <div id="ghl-utils-progress-bar" style="
            height: 100%;
            background: linear-gradient(90deg, #FB8500 0%, #FF9500 100%);
            width: 0%;
            transition: width 0.3s ease;
          "></div>
        </div>
        <p id="ghl-utils-progress-text" style="
          margin: 8px 0 0;
          text-align: center;
          font-size: 13px;
          color: #6b7280;
        ">Processing...</p>
      </div>
    </div>
  `;

  overlay.appendChild(modal);
  document.body.appendChild(overlay);

  // Add event listeners
  document.getElementById('ghl-utils-close-modal').addEventListener('click', closeBulkEditModal);
  document.getElementById('ghl-utils-cancel-btn').addEventListener('click', closeBulkEditModal);
  document.getElementById('ghl-utils-apply-btn').addEventListener('click', applyBulkEdit);

  // Show/hide value input based on action
  document.getElementById('ghl-utils-action-select').addEventListener('change', (e) => {
    const valueContainer = document.getElementById('ghl-utils-value-container');
    if (e.target.value === 'clear') {
      valueContainer.style.display = 'none';
    } else {
      valueContainer.style.display = 'block';
    }
  });
}

// Close bulk edit modal
function closeBulkEditModal() {
  const modal = document.getElementById('ghl-utils-bulk-edit-modal');
  if (modal) {
    modal.remove();
  }
  bulkEditorState.isProcessing = false;
}

// Fetch custom fields from GHL API
async function fetchCustomFields() {
  try {
    // Check if API is available
    if (!window.ghlUtilsAPI) {
      console.error('[Bulk Editor] API not available!');
      throw new Error('API not initialized');
    }

    // Wait for API to be ready
    console.log('[Bulk Editor] Waiting for API to be ready...');
    await window.ghlUtilsAPI.waitForReady();
    console.log('[Bulk Editor] API is ready!');

    // Get location ID if not already set
    if (!bulkEditorState.locationId) {
      bulkEditorState.locationId = window.ghlUtilsAPI.getLocationId();
    }

    if (!bulkEditorState.locationId) {
      console.warn('[Bulk Editor] Missing location ID');
      return;
    }

    console.log('[Bulk Editor] Fetching custom fields for location:', bulkEditorState.locationId);

    const response = await window.ghlUtilsAPI.get(`/locations/${bulkEditorState.locationId}/customFields`);

    if (response.status !== 200) {
      throw new Error(`API error: ${response.status}`);
    }

    console.log('[Bulk Editor] Custom fields response:', response.data);

    // Handle different response structures
    if (response.data.customFields) {
      bulkEditorState.customFields = response.data.customFields;
    } else if (Array.isArray(response.data)) {
      bulkEditorState.customFields = response.data;
    } else {
      bulkEditorState.customFields = [];
    }

    console.log('[Bulk Editor] Custom fields loaded:', bulkEditorState.customFields.length, bulkEditorState.customFields);
  } catch (error) {
    console.error('[Bulk Editor] Error fetching custom fields:', error);
    if (error.message.includes('not available')) {
      showToast('⚠️ Please refresh the page and try again.', 4000, 'error');
    } else {
      showToast('⚠️ Error loading custom fields. Please try again.', 4000, 'error');
    }
  }
}

// Apply bulk edit
async function applyBulkEdit() {
  const field = document.getElementById('ghl-utils-field-select').value;
  const action = document.getElementById('ghl-utils-action-select').value;
  const newValue = document.getElementById('ghl-utils-new-value').value;

  // Validation
  if (!field) {
    showToast('⚠️ Please select a field to edit', 3000, 'warning');
    return;
  }

  if (action !== 'clear' && !newValue.trim()) {
    showToast('⚠️ Please enter a new value', 3000, 'warning');
    return;
  }

  // Disable buttons
  bulkEditorState.isProcessing = true;
  document.getElementById('ghl-utils-apply-btn').disabled = true;
  document.getElementById('ghl-utils-apply-btn').style.opacity = '0.6';
  document.getElementById('ghl-utils-apply-btn').style.cursor = 'not-allowed';

  // Show progress
  document.getElementById('ghl-utils-progress-container').style.display = 'block';

  // Process updates with rate limiting
  const results = await processBulkUpdate(field, action, newValue);

  // Show results
  const successCount = results.filter(r => r.success).length;
  const failCount = results.filter(r => !r.success).length;

  if (failCount === 0) {
    showToast(`✅ Successfully updated ${successCount} contacts!`, 4000, 'success');
  } else {
    showToast(`⚠️ Updated ${successCount} contacts, ${failCount} failed. Check console for details.`, 5000, 'warning');
    console.error('[GHL Utils] Failed updates:', results.filter(r => !r.success));
  }

  // Close modal
  setTimeout(() => {
    closeBulkEditModal();
    // Clear selection
    bulkEditorState.selectedContacts = [];
    hideBulkEditButton();
  }, 2000);
}

// Process bulk update with rate limiting
async function processBulkUpdate(field, action, newValue) {
  const results = [];
  const total = bulkEditorState.selectedContacts.length;
  const batchSize = 5; // Process 5 at a time to avoid rate limits
  const delayBetweenBatches = 1000; // 1 second delay between batches

  for (let i = 0; i < total; i += batchSize) {
    const batch = bulkEditorState.selectedContacts.slice(i, i + batchSize);
    const batchPromises = batch.map(contactId =>
      updateContact(contactId, field, action, newValue)
    );

    const batchResults = await Promise.all(batchPromises);
    results.push(...batchResults);

    // Update progress
    const progress = Math.round(((i + batch.length) / total) * 100);
    updateProgress(progress, `Updated ${i + batch.length} of ${total} contacts...`);

    // Delay before next batch (except for last batch)
    if (i + batchSize < total) {
      await new Promise(resolve => setTimeout(resolve, delayBetweenBatches));
    }
  }

  return results;
}

// Update single contact via GHL API
async function updateContact(contactId, field, action, newValue) {
  try {
    // Build update payload
    const payload = {};

    if (field.startsWith('customField.')) {
      const customFieldId = field.replace('customField.', '');
      payload.customFields = [{
        id: customFieldId,
        value: action === 'clear' ? '' : newValue
      }];
    } else if (field === 'tags') {
      if (action === 'clear') {
        payload.tags = [];
      } else {
        // Split comma-separated tags
        const tags = newValue.split(',').map(t => t.trim()).filter(t => t);
        payload.tags = tags;
      }
    } else {
      // Standard field
      payload[field] = action === 'clear' ? '' : newValue;
    }

    const response = await window.ghlUtilsAPI.put(`/contacts/${contactId}`, payload);

    if (response.status !== 200) {
      throw new Error(`API error ${response.status}`);
    }

    return { success: true, contactId, data: response.data };
  } catch (error) {
    console.error(`[GHL Utils] Error updating contact ${contactId}:`, error);
    return { success: false, contactId, error: error.message };
  }
}

// Update progress bar
function updateProgress(percent, text) {
  const progressBar = document.getElementById('ghl-utils-progress-bar');
  const progressText = document.getElementById('ghl-utils-progress-text');

  if (progressBar) {
    progressBar.style.width = `${percent}%`;
  }

  if (progressText) {
    progressText.textContent = text;
  }
}

// Show toast notification (reuse from main content.js if available, otherwise create inline)
function showToast(message, duration = 3000, type = 'info') {
  // Try to use existing showToast function from content.js (not this one!)
  if (typeof window.ghlUtilsShowToast === 'function') {
    window.ghlUtilsShowToast(message);
    return;
  }

  // Create toast inline
  const toast = document.createElement('div');
  const colors = {
    success: '#10b981',
    warning: '#f59e0b',
    error: '#ef4444',
    info: '#3b82f6'
  };

  toast.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: ${colors[type] || colors.info};
    color: white;
    padding: 12px 20px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
    z-index: 999999999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    font-size: 14px;
    max-width: 400px;
    transform: translateX(400px);
    opacity: 0;
    transition: all 0.3s ease-out;
  `;
  toast.textContent = message;

  document.body.appendChild(toast);

  // Trigger animation
  requestAnimationFrame(() => {
    requestAnimationFrame(() => {
      toast.style.transform = 'translateX(0)';
      toast.style.opacity = '1';
    });
  });

  // Auto-hide
  setTimeout(() => {
    toast.style.transform = 'translateX(400px)';
    toast.style.opacity = '0';
    setTimeout(() => {
      if (toast.parentNode) {
        toast.remove();
      }
    }, 300);
  }, duration);
}

// Initialize when content script loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initBulkContactEditor);
} else {
  initBulkContactEditor();
}

// Re-initialize on URL changes (SPA navigation)
let lastBulkEditorUrl = location.href;
new MutationObserver(() => {
  if (location.href !== lastBulkEditorUrl) {
    lastBulkEditorUrl = location.href;
    setTimeout(initBulkContactEditor, 1000);
  }
}).observe(document.querySelector('head > title') || document.body, {
  subtree: true,
  characterData: true,
  childList: true
});
