// GHL Utils - Contact Field Inspector & Validator
// Shows all contact fields in a single pane with validation status

// State management
let fieldInspectorState = {
  enabled: true,
  isOpen: false,
  currentContactId: null,
  contactData: null,
  customFields: [],
  locationId: null,
  searchQuery: '',
  showFilledOnly: false
};

// Standard contact fields to check
const STANDARD_FIELDS = [
  { key: 'firstName', label: 'First Name', required: false, category: 'Basic Info' },
  { key: 'lastName', label: 'Last Name', required: false, category: 'Basic Info' },
  { key: 'name', label: 'Full Name', required: false, category: 'Basic Info' },
  { key: 'email', label: 'Email', required: true, category: 'Contact Info' },
  { key: 'phone', label: 'Phone', required: false, category: 'Contact Info' },
  { key: 'address1', label: 'Address', required: false, category: 'Location' },
  { key: 'city', label: 'City', required: false, category: 'Location' },
  { key: 'state', label: 'State', required: false, category: 'Location' },
  { key: 'postalCode', label: 'Postal Code', required: false, category: 'Location' },
  { key: 'country', label: 'Country', required: false, category: 'Location' },
  { key: 'website', label: 'Website', required: false, category: 'Other' },
  { key: 'dateOfBirth', label: 'Date of Birth', required: false, category: 'Other' },
  { key: 'companyName', label: 'Company Name', required: false, category: 'Business' },
  { key: 'source', label: 'Source', required: false, category: 'Tracking' },
  { key: 'tags', label: 'Tags', required: false, category: 'Organization' }
];

// Initialize field inspector
async function initFieldInspector() {
  // Check if Chrome API is available
  if (typeof chrome === 'undefined' || !chrome.storage) {
    console.error('[Field Inspector] Chrome API not available');
    return;
  }

  try {
    // Load settings
    const stored = await chrome.storage.sync.get(['fieldInspectorEnabled', 'locationId']);
    fieldInspectorState.enabled = stored.fieldInspectorEnabled !== false;
    fieldInspectorState.locationId = stored.locationId;
  } catch (error) {
    console.error('[Field Inspector] Error loading settings:', error);
    // Continue with defaults
    fieldInspectorState.enabled = true;
  }

  if (!fieldInspectorState.enabled) {
    return;
  }

  // Check if on contact detail page - only show floating button here
  if (isContactDetailPage()) {
    addInspectorButton();
    watchForContactChanges();
  }

  // Check if on contacts list page - only add eye icons, no floating button
  if (isContactsListPage()) {
    addListViewClickHandlers();
  }

  // Watch for navigation to contact pages
  watchForContactPageNavigation();
}

// Check if current page is a contact detail page
function isContactDetailPage() {
  const url = window.location.href;
  return url.includes('/contacts/detail/');
}

// Check if current page is contacts list page
function isContactsListPage() {
  const url = window.location.href;
  return url.includes('/contacts') && !url.includes('/contacts/detail/');
}

// Extract contact ID from current page
function extractContactId() {
  const url = window.location.href;
  const match = url.match(/\/contacts\/(?:detail|smart_list)\/([A-Za-z0-9_-]+)/);
  return match ? match[1] : null;
}

// Add inspector button to page
function addInspectorButton() {
  // Remove existing button if any
  const existing = document.getElementById('ghl-utils-field-inspector-btn');
  if (existing) existing.remove();

  // Create floating button
  const button = document.createElement('button');
  button.id = 'ghl-utils-field-inspector-btn';
  button.innerHTML = `
    <svg width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
      <path d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
      <path d="M9 12h6m-6 4h6"/>
    </svg>
    <span style="margin-left: 8px;">Inspect Fields</span>
  `;
  button.style.cssText = `
    position: fixed;
    bottom: 24px;
    right: 24px;
    z-index: 999998;
    padding: 12px 20px;
    background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
    color: white;
    border: none;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 500;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    cursor: pointer;
    box-shadow: 0 4px 16px rgba(251, 133, 0, 0.4);
    transition: all 0.2s ease;
    display: flex;
    align-items: center;
  `;

  // Adjust position if bulk edit button exists
  const checkBulkEditButton = () => {
    const bulkEditBtn = document.getElementById('ghl-utils-bulk-edit-btn');
    if (bulkEditBtn) {
      button.style.bottom = '90px'; // Position above bulk edit button
    } else {
      button.style.bottom = '24px';
    }
  };

  checkBulkEditButton();
  // Keep checking in case bulk edit button appears
  setInterval(checkBulkEditButton, 1000);

  // Hover effects
  button.addEventListener('mouseenter', () => {
    button.style.transform = 'translateY(-2px)';
    button.style.boxShadow = '0 6px 20px rgba(251, 133, 0, 0.5)';
  });

  button.addEventListener('mouseleave', () => {
    button.style.transform = 'translateY(0)';
    button.style.boxShadow = '0 4px 16px rgba(251, 133, 0, 0.4)';
  });

  button.addEventListener('click', openFieldInspector);

  document.body.appendChild(button);
}

// Add click handlers to contact rows in list view
function addListViewClickHandlers() {
  // Wait for page to load
  setTimeout(() => {
    setupListViewClickHandlers();
  }, 2000);

  // Re-setup on DOM changes (for SPA navigation and lazy loading)
  const observer = new MutationObserver(() => {
    setupListViewClickHandlers();
  });

  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

function setupListViewClickHandlers() {
  // Find contact rows - they typically have contact IDs in their attributes or children
  const rows = document.querySelectorAll('tr[id]');

  rows.forEach(row => {
    // Skip if already has handler
    if (row.hasAttribute('data-inspector-handler')) {
      return;
    }

    // Skip if not a contact row (check for checkbox with contact ID)
    const checkbox = row.querySelector('input[type="checkbox"][id]');
    if (!checkbox) {
      return;
    }

    row.setAttribute('data-inspector-handler', 'true');

    // Add eye icon button directly in the row (next to checkbox)
    const checkboxCell = checkbox.closest('td');
    if (checkboxCell && !checkboxCell.querySelector('.ghl-utils-eye-icon')) {
      const eyeIcon = document.createElement('button');
      eyeIcon.className = 'ghl-utils-eye-icon';
      eyeIcon.title = 'Inspect Fields';
      eyeIcon.style.cssText = `
        display: inline-flex;
        align-items: center;
        justify-content: center;
        width: 28px;
        height: 28px;
        margin-left: 8px;
        padding: 0;
        background: transparent;
        border: 1px solid #d1d5db;
        border-radius: 6px;
        cursor: pointer;
        transition: all 0.2s ease;
        vertical-align: middle;
      `;

      eyeIcon.innerHTML = `
        <svg width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="transition: stroke 0.2s ease;">
          <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
          <circle cx="12" cy="12" r="3"/>
        </svg>
      `;

      eyeIcon.addEventListener('click', (e) => {
        e.preventDefault();
        e.stopPropagation();

        // Try to find contact ID from the row
        const contactId = findContactIdFromListRow(row);
        if (contactId) {
          openFieldInspectorForContact(contactId);
        } else {
          showToast('⚠️ Could not detect contact ID from this row');
        }
      });

      // Add orange hover effect
      const svgIcon = eyeIcon.querySelector('svg');
      eyeIcon.addEventListener('mouseenter', () => {
        eyeIcon.style.background = 'linear-gradient(135deg, #FB8500 0%, #FF9500 100%)';
        eyeIcon.style.borderColor = '#FB8500';
        if (svgIcon) svgIcon.style.stroke = 'white';
      });

      eyeIcon.addEventListener('mouseleave', () => {
        eyeIcon.style.background = 'transparent';
        eyeIcon.style.borderColor = '#d1d5db';
        if (svgIcon) svgIcon.style.stroke = 'currentColor';
      });

      // Insert after checkbox
      const label = checkboxCell.querySelector('label');
      if (label) {
        label.parentNode.insertBefore(eyeIcon, label.nextSibling);
      } else {
        checkbox.parentNode.insertBefore(eyeIcon, checkbox.nextSibling);
      }
    }

    // Find the dropdown menu in this row
    const dropdownMenu = row.querySelector('.dropdown-menu');
    if (dropdownMenu) {
      // Check if we already added the button
      if (!dropdownMenu.querySelector('.ghl-utils-inspect-fields-btn')) {
        // Add "Inspect Fields" option to dropdown
        const inspectBtn = document.createElement('a');
        inspectBtn.className = 'mx-1 my-1 pointer dropdown-item ghl-utils-inspect-fields-btn';
        inspectBtn.href = 'javascript:void(0);';
        inspectBtn.style.cssText = `
          transition: all 0.2s ease;
          border-radius: 4px;
        `;
        inspectBtn.innerHTML = `
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display: inline-block; vertical-align: middle; margin-right: 6px; transition: stroke 0.2s ease;">
            <path d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2M9 5a2 2 0 002 2h2a2 2 0 002-2M9 5a2 2 0 012-2h2a2 2 0 012 2"/>
            <path d="M9 12h6m-6 4h6"/>
          </svg>
          <span style="font-weight: 500;">Inspect Fields</span>
        `;

        inspectBtn.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();

          // Try to find contact ID from the row
          const contactId = findContactIdFromListRow(row);
          if (contactId) {
            openFieldInspectorForContact(contactId);
          } else {
            showToast('⚠️ Could not detect contact ID from this row');
          }
        });

        // Add orange hover effect
        const svgIcon = inspectBtn.querySelector('svg');
        inspectBtn.addEventListener('mouseenter', () => {
          inspectBtn.style.background = 'linear-gradient(135deg, #FB8500 0%, #FF9500 100%)';
          inspectBtn.style.color = 'white';
          if (svgIcon) svgIcon.style.stroke = 'white';
        });

        inspectBtn.addEventListener('mouseleave', () => {
          inspectBtn.style.background = '';
          inspectBtn.style.color = '';
          if (svgIcon) svgIcon.style.stroke = 'currentColor';
        });

        // Add as first item in dropdown
        dropdownMenu.insertBefore(inspectBtn, dropdownMenu.firstChild);

        // Add Audit Log button after Inspect Fields
        const auditBtn = document.createElement('a');
        auditBtn.className = 'mx-1 my-1 pointer dropdown-item ghl-utils-audit-log-btn';
        auditBtn.href = 'javascript:void(0);';
        auditBtn.style.cssText = `
          transition: all 0.2s ease;
          border-radius: 4px;
        `;
        auditBtn.innerHTML = `
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" style="display: inline-block; vertical-align: middle; margin-right: 6px; transition: stroke 0.2s ease;">
            <path d="M9 5H7a2 2 0 00-2 2v12a2 2 0 002 2h10a2 2 0 002-2V7a2 2 0 00-2-2h-2"/>
            <rect x="9" y="3" width="6" height="4" rx="1"/>
            <path d="M9 12h6M9 16h6"/>
          </svg>
          <span style="font-weight: 500;">Audit Log of Contact</span>
        `;

        auditBtn.addEventListener('click', (e) => {
          e.preventDefault();
          e.stopPropagation();

          // Try to find contact ID from the row
          const contactId = findContactIdFromListRow(row);
          if (contactId) {
            // Get location ID
            const locationMatch = window.location.href.match(/\/location\/([A-Za-z0-9_-]{18,28})/);
            if (locationMatch) {
              const locationId = locationMatch[1];
              const auditUrl = `https://app.gohighlevel.com/v2/location/${locationId}/settings/audit/logs?documentId=${contactId}`;
              window.open(auditUrl, '_blank');
            } else {
              showToast('⚠️ Could not detect location ID');
            }
          } else {
            showToast('⚠️ Could not detect contact ID from this row');
          }
        });

        // Add orange hover effect
        const auditSvgIcon = auditBtn.querySelector('svg');
        auditBtn.addEventListener('mouseenter', () => {
          auditBtn.style.background = 'linear-gradient(135deg, #FB8500 0%, #FF9500 100%)';
          auditBtn.style.color = 'white';
          if (auditSvgIcon) auditSvgIcon.style.stroke = 'white';
        });

        auditBtn.addEventListener('mouseleave', () => {
          auditBtn.style.background = '';
          auditBtn.style.color = '';
          if (auditSvgIcon) auditSvgIcon.style.stroke = 'currentColor';
        });

        dropdownMenu.insertBefore(auditBtn, inspectBtn.nextSibling);

        // Add separator after our buttons
        const separator = document.createElement('div');
        separator.className = 'dropdown-divider';
        separator.style.cssText = 'margin: 4px 0; border-top: 1px solid #e5e7eb;';
        dropdownMenu.insertBefore(separator, auditBtn.nextSibling);
      }
    }
  });
}

// Find contact ID from a list view row
function findContactIdFromListRow(row) {
  // Strategy 1: Check row ID attribute (most common in GHL list views)
  let rowId = row.getAttribute('id');
  if (rowId && isValidContactId(rowId)) {
    return rowId;
  }

  // Strategy 2: Check checkbox ID (checkbox ID is usually the contact ID)
  const checkbox = row.querySelector('input[type="checkbox"][id]');
  if (checkbox) {
    const checkboxId = checkbox.getAttribute('id');
    if (checkboxId && isValidContactId(checkboxId)) {
      return checkboxId;
    }
  }

  // Strategy 3: Look for contact detail links
  const detailLink = row.querySelector('a[href*="/contacts/detail/"]');
  if (detailLink) {
    const match = detailLink.href.match(/\/contacts\/detail\/([A-Za-z0-9_-]{20,30})/);
    if (match && isValidContactId(match[1])) {
      return match[1];
    }
  }

  // Strategy 4: Check row data attributes
  let contactId = row.getAttribute('data-contact-id') ||
                  row.getAttribute('data-id');

  if (contactId && isValidContactId(contactId)) {
    return contactId.replace(/^(contact-|id-|row-)/, '');
  }

  // Strategy 5: Check child elements with data-contact-id
  const contactElement = row.querySelector('[data-contact-id]');
  if (contactElement) {
    contactId = contactElement.getAttribute('data-contact-id');
    if (contactId && isValidContactId(contactId)) {
      return contactId;
    }
  }

  return null;
}

// Validate if a string is a valid GHL contact ID
function isValidContactId(id) {
  if (!id || typeof id !== 'string') return false;

  // Clean up common prefixes
  const cleanId = id.replace(/^(contact-|id-|row-)/, '');

  // GHL contact IDs are typically 20-30 characters, alphanumeric with dashes/underscores
  if (cleanId.length < 20 || cleanId.length > 30) return false;

  // Should only contain valid characters
  if (!/^[A-Za-z0-9_-]+$/.test(cleanId)) return false;

  // Should not be all numbers (likely not a contact ID)
  if (/^\d+$/.test(cleanId)) return false;

  return true;
}

// Open field inspector for a specific contact ID
async function openFieldInspectorForContact(contactId) {
  fieldInspectorState.currentContactId = contactId;
  await openFieldInspector();
}

// Open field inspector panel
async function openFieldInspector() {
  if (fieldInspectorState.isOpen) {
    closeFieldInspector();
    return;
  }

  fieldInspectorState.isOpen = true;
  fieldInspectorState.searchQuery = ''; // Reset search

  // If we don't have a contact ID yet, try to extract it
  if (!fieldInspectorState.currentContactId) {
    fieldInspectorState.currentContactId = extractContactId();
  }

  if (!fieldInspectorState.currentContactId) {
    showToast('⚠️ Could not detect contact ID');
    fieldInspectorState.isOpen = false;
    return;
  }

  // Create overlay
  const overlay = document.createElement('div');
  overlay.id = 'ghl-utils-field-inspector-overlay';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    z-index: 9999999;
    animation: fadeIn 0.2s ease-out;
  `;

  overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
      closeFieldInspector();
    }
  });

  // Create side panel
  const panel = document.createElement('div');
  panel.id = 'ghl-utils-field-inspector-panel';
  panel.style.cssText = `
    position: fixed;
    top: 0;
    right: 0;
    bottom: 0;
    width: 500px;
    max-width: 90vw;
    background: white;
    box-shadow: -4px 0 24px rgba(0, 0, 0, 0.2);
    display: flex;
    flex-direction: column;
    animation: slideInRight 0.3s ease-out;
    z-index: 10000000;
  `;

  panel.innerHTML = `
    <div style="
      padding: 12px 16px;
      background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
      color: white;
      display: flex;
      justify-content: space-between;
      align-items: center;
    ">
      <div>
        <h2 style="margin: 0; font-size: 16px; font-weight: 600;">Contact Fields</h2>
        <p style="margin: 2px 0 0; font-size: 12px; opacity: 0.9;">Loading...</p>
      </div>
      <button id="ghl-utils-close-inspector" style="
        background: rgba(255, 255, 255, 0.2);
        border: none;
        color: white;
        font-size: 22px;
        cursor: pointer;
        width: 28px;
        height: 28px;
        border-radius: 4px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: background 0.2s;
      ">&times;</button>
    </div>

    <div style="padding: 12px 12px 0;">
      <input
        type="text"
        id="ghl-utils-field-search"
        placeholder="Search fields..."
        style="
          width: 100%;
          padding: 8px 12px;
          border: 1px solid #e5e7eb;
          border-radius: 6px;
          font-size: 13px;
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          outline: none;
          transition: border-color 0.2s;
        "
      />
      <label style="
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-top: 10px;
        cursor: pointer;
        user-select: none;
        padding: 8px 10px;
        background: #f9fafb;
        border-radius: 6px;
        transition: background 0.2s;
      " onmouseover="this.style.background='#f3f4f6'" onmouseout="this.style.background='#f9fafb'">
        <span style="
          font-size: 13px;
          color: #374151;
          font-weight: 500;
        ">Show filled fields only</span>
        <div style="position: relative;">
          <input
            type="checkbox"
            id="ghl-utils-show-filled-only"
            style="
              position: absolute;
              opacity: 0;
              width: 0;
              height: 0;
            "
          />
          <div class="toggle-switch" style="
            width: 44px;
            height: 24px;
            background: #d1d5db;
            border-radius: 12px;
            position: relative;
            transition: background 0.3s;
          ">
            <div class="toggle-slider" style="
              position: absolute;
              top: 2px;
              left: 2px;
              width: 20px;
              height: 20px;
              background: white;
              border-radius: 50%;
              transition: transform 0.3s;
              box-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
            "></div>
          </div>
        </div>
      </label>
    </div>

    <div id="ghl-utils-field-inspector-content" style="
      flex: 1;
      overflow-y: auto;
      padding: 12px;
    ">
      <div style="text-align: center; padding: 30px 16px; color: #6b7280;">
        <div style="
          width: 32px;
          height: 32px;
          border: 3px solid #FB8500;
          border-radius: 50%;
          border-top-color: transparent;
          animation: spin 1s linear infinite;
          margin: 0 auto 12px;
        "></div>
        <p style="font-size: 13px; margin: 0;">Loading fields...</p>
      </div>
    </div>

    <style>
      @keyframes fadeIn {
        from { opacity: 0; }
        to { opacity: 1; }
      }
      @keyframes slideInRight {
        from { transform: translateX(100%); }
        to { transform: translateX(0); }
      }
      @keyframes spin {
        to { transform: rotate(360deg); }
      }
    </style>
  `;

  overlay.appendChild(panel);
  document.body.appendChild(overlay);

  // Close button handler
  document.getElementById('ghl-utils-close-inspector').addEventListener('click', closeFieldInspector);

  // Search input handler
  const searchInput = document.getElementById('ghl-utils-field-search');
  searchInput.addEventListener('input', (e) => {
    fieldInspectorState.searchQuery = e.target.value.toLowerCase();
    displayContactFields();
  });

  // Focus and border effect for search
  searchInput.addEventListener('focus', () => {
    searchInput.style.borderColor = '#FB8500';
  });
  searchInput.addEventListener('blur', () => {
    searchInput.style.borderColor = '#e5e7eb';
  });

  // Show filled only toggle handler
  const showFilledOnlyCheckbox = document.getElementById('ghl-utils-show-filled-only');
  const toggleSwitch = document.querySelector('.toggle-switch');
  const toggleSlider = document.querySelector('.toggle-slider');

  // Function to update toggle visual state
  const updateToggleState = (isChecked) => {
    if (isChecked) {
      toggleSwitch.style.background = 'linear-gradient(135deg, #FB8500 0%, #FF9500 100%)';
      toggleSlider.style.transform = 'translateX(20px)';
    } else {
      toggleSwitch.style.background = '#d1d5db';
      toggleSlider.style.transform = 'translateX(0)';
    }
  };

  showFilledOnlyCheckbox.addEventListener('change', (e) => {
    fieldInspectorState.showFilledOnly = e.target.checked;
    updateToggleState(e.target.checked);
    displayContactFields();
    // Save preference
    try {
      if (chrome && chrome.storage) {
        chrome.storage.sync.set({ showFilledFieldsOnly: e.target.checked });
      }
    } catch (error) {
      console.warn('[Field Inspector] Could not save preference:', error);
    }
  });

  // Load saved preference before fetching data
  try {
    if (chrome && chrome.storage) {
      const savedPrefs = await chrome.storage.sync.get(['showFilledFieldsOnly']);
      if (savedPrefs.showFilledFieldsOnly !== undefined) {
        fieldInspectorState.showFilledOnly = savedPrefs.showFilledFieldsOnly;
        showFilledOnlyCheckbox.checked = savedPrefs.showFilledFieldsOnly;
        updateToggleState(savedPrefs.showFilledFieldsOnly);
      }
    }
  } catch (error) {
    console.warn('[Field Inspector] Could not load saved preference:', error);
  }

  // Fetch and display contact data
  await fetchAndDisplayContactData();
}

// Close field inspector
function closeFieldInspector() {
  const overlay = document.getElementById('ghl-utils-field-inspector-overlay');
  if (overlay) {
    overlay.remove();
  }
  fieldInspectorState.isOpen = false;
  fieldInspectorState.currentContactId = null; // Clear for next use
  fieldInspectorState.searchQuery = ''; // Clear search
  // Note: we keep showFilledOnly state so it persists across sessions
}

// Fetch and display contact data
async function fetchAndDisplayContactData() {
  const content = document.getElementById('ghl-utils-field-inspector-content');
  if (!content) return;

  console.log('[Field Inspector] Starting to fetch contact data for ID:', fieldInspectorState.currentContactId);

  try {
    // Check if API is available
    if (!window.ghlUtilsAPI) {
      console.error('[Field Inspector] API not available!');
      throw new Error('API not initialized');
    }

    // Wait for API to be ready (silently)
    await window.ghlUtilsAPI.waitForReady();

    // Fetch contact data
    const response = await window.ghlUtilsAPI.get(`/contacts/${fieldInspectorState.currentContactId}`);

    console.log('[Field Inspector] API response received:', response.status);

    if (response.status !== 200) {
      console.error('[Field Inspector] API error response:', response);
      throw new Error(`API error: ${response.status}`);
    }

    const data = response.data;
    console.log('[Field Inspector] Contact data received:', data);
    fieldInspectorState.contactData = data.contact || data;

    // Fetch custom fields
    await fetchCustomFields();

    // Display the data
    displayContactFields();

  } catch (error) {
    let errorMessage = error.message;
    let helpText = '';

    if (error.message.includes('Not authenticated')) {
      errorMessage = 'Not logged into GHL';
      helpText = 'Please make sure you are logged into GoHighLevel.';
    } else if (error.message.includes('400')) {
      errorMessage = 'Invalid contact ID';
      helpText = 'The contact ID may be incorrect. Try opening the contact detail page first.';
    } else if (error.message.includes('401')) {
      errorMessage = 'Session expired';
      helpText = 'Please refresh the page and try again.';
    } else if (error.message.includes('404')) {
      errorMessage = 'Contact not found';
      helpText = 'This contact may have been deleted.';
    } else if (error.message.includes('403')) {
      errorMessage = 'Permission denied';
      helpText = 'You may not have access to this contact.';
    }

    content.innerHTML = `
      <div style="text-align: center; padding: 30px 16px;">
        <div style="font-size: 40px; margin-bottom: 12px;">⚠️</div>
        <h3 style="margin: 0 0 6px; font-size: 14px; color: #111827;">Error Loading Contact</h3>
        <p style="margin: 0 0 6px; font-size: 12px; color: #6b7280;">
          ${errorMessage}
        </p>
        ${helpText ? `<p style="margin: 0; font-size: 11px; color: #9ca3af;">${helpText}</p>` : ''}
      </div>
    `;
  }
}

// Fetch custom fields
async function fetchCustomFields() {
  try {
    // Get location ID if not already set
    if (!fieldInspectorState.locationId) {
      fieldInspectorState.locationId = window.ghlUtilsAPI.getLocationId();
    }

    if (!fieldInspectorState.locationId) {
      console.warn('[Field Inspector] No location ID, skipping custom fields');
      return; // Can't fetch without location ID
    }

    console.log('[Field Inspector] Fetching custom fields for location:', fieldInspectorState.locationId);

    const response = await window.ghlUtilsAPI.get(`/locations/${fieldInspectorState.locationId}/customFields`);

    if (response.status === 200 && response.data) {
      fieldInspectorState.customFields = response.data.customFields || [];
      console.log('[Field Inspector] Custom fields loaded:', fieldInspectorState.customFields.length);
    }
  } catch (error) {
    console.warn('[Field Inspector] Failed to fetch custom fields (optional):', error);
    // Silently fail, custom fields are optional
  }
}

// Display contact fields
function displayContactFields() {
  const content = document.getElementById('ghl-utils-field-inspector-content');
  if (!content || !fieldInspectorState.contactData) return;

  const contact = fieldInspectorState.contactData;

  // Debug: Log contact structure to help identify custom field location
  console.log('[Field Inspector] Contact data structure:', {
    hasCustomField: !!contact.customField,
    customFieldType: typeof contact.customField,
    hasCustomFields: !!contact.customFields,
    customFieldsIsArray: Array.isArray(contact.customFields),
    customFieldsLength: Array.isArray(contact.customFields) ? contact.customFields.length : 0,
    sampleCustomFieldKeys: contact.customField ? Object.keys(contact.customField).slice(0, 3) : [],
    sampleCustomFieldsData: Array.isArray(contact.customFields) ? contact.customFields.slice(0, 2) : []
  });

  // Calculate data quality score
  const score = calculateDataQuality(contact);

  // Update header - will be updated again after filtering if search is active
  const headerSubtitle = document.querySelector('#ghl-utils-field-inspector-panel p');
  if (headerSubtitle) {
    headerSubtitle.textContent = `${score.percentage}% complete • ${score.filled}/${score.total} fields`;
    headerSubtitle.setAttribute('data-base-text', headerSubtitle.textContent);
  }

  // Build field categories
  const categories = {};

  // Standard fields
  STANDARD_FIELDS.forEach(field => {
    if (!categories[field.category]) {
      categories[field.category] = [];
    }

    const value = contact[field.key];
    const status = getFieldStatus(value, field.required);

    categories[field.category].push({
      label: field.label,
      key: field.key,
      value: value,
      status: status,
      required: field.required,
      type: 'standard'
    });
  });

  // Custom fields
  if (fieldInspectorState.customFields.length > 0) {
    categories['Custom Fields'] = [];

    fieldInspectorState.customFields.forEach(customField => {
      // Try multiple possible locations for custom field values
      let value = null;
      let foundMethod = null;

      // Method 1: customField object (singular)
      if (contact.customField && typeof contact.customField === 'object') {
        value = contact.customField[customField.id];
        if (value) foundMethod = 'customField[id]';
      }

      // Method 2: customFields array
      if (!value && Array.isArray(contact.customFields)) {
        const customFieldData = contact.customFields.find(cf => cf.id === customField.id);
        value = customFieldData?.value;
        if (value) foundMethod = 'customFields array';
      }

      // Method 3: Direct property on contact object (some GHL endpoints do this)
      if (!value && contact[customField.id]) {
        value = contact[customField.id];
        if (value) foundMethod = 'direct property';
      }

      console.log(`[Field Inspector] Custom field "${customField.name}" (${customField.id}):`, {
        value: value,
        foundMethod: foundMethod,
        checkedMethods: {
          hasCustomField: !!contact.customField,
          hasCustomFields: Array.isArray(contact.customFields),
          hasDirectProperty: !!contact[customField.id]
        }
      });

      const status = getFieldStatus(value, false);

      categories['Custom Fields'].push({
        label: customField.name,
        key: customField.id,
        value: value,
        status: status,
        required: false,
        type: 'custom'
      });
    });
  }

  // Render categories
  let html = `
    <div style="
      background: linear-gradient(135deg, #fef3c7 0%, #fde68a 100%);
      border: 1px solid #fbbf24;
      border-radius: 6px;
      padding: 10px 12px;
      margin-bottom: 12px;
      display: flex;
      align-items: center;
      gap: 10px;
    ">
      <div style="
        width: 42px;
        height: 42px;
        border-radius: 50%;
        background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
        font-weight: 700;
        color: white;
        flex-shrink: 0;
      ">${score.percentage}%</div>
      <div style="flex: 1; min-width: 0;">
        <div style="font-size: 12px; font-weight: 600; color: #78350f;">
          Data Quality: ${score.filled}/${score.total} fields
        </div>
      </div>
    </div>
  `;

  // Filter fields based on search query and filled status
  const searchQuery = fieldInspectorState.searchQuery;
  const showFilledOnly = fieldInspectorState.showFilledOnly;
  let totalFilteredFields = 0;

  // Render each category
  Object.keys(categories).forEach(categoryName => {
    const fields = categories[categoryName];

    // Filter fields based on search query and filled status
    let filteredFields = fields;

    // Apply search filter
    if (searchQuery) {
      filteredFields = filteredFields.filter(field =>
        field.label.toLowerCase().includes(searchQuery) ||
        (field.value && String(field.value).toLowerCase().includes(searchQuery))
      );
    }

    // Apply filled-only filter
    if (showFilledOnly) {
      filteredFields = filteredFields.filter(field => {
        const value = field.value;

        // Check if field has a value (not null, undefined, or empty string/array)
        if (value === null || value === undefined || value === '') {
          return false;
        }

        // Check for empty arrays
        if (Array.isArray(value) && value.length === 0) {
          return false;
        }

        // Check for string representations of empty values
        if (typeof value === 'string') {
          const trimmed = value.trim();
          if (trimmed === '' || trimmed === 'null' || trimmed === 'undefined' || trimmed === 'N/A') {
            return false;
          }
        }

        return true;
      });
    }

    // Skip category if no fields match
    if (filteredFields.length === 0) {
      return;
    }

    totalFilteredFields += filteredFields.length;

    html += `
      <div style="margin-bottom: 16px;">
        <h3 style="
          font-size: 11px;
          font-weight: 600;
          color: #6b7280;
          text-transform: uppercase;
          letter-spacing: 0.5px;
          margin: 0 0 6px;
          padding-left: 2px;
        ">${categoryName}</h3>
    `;

    filteredFields.forEach(field => {
      const statusIcon = field.status === 'valid' ? '✓' :
                        field.status === 'missing' ? '✗' : '⚠';
      const statusColor = field.status === 'valid' ? '#10b981' :
                         field.status === 'missing' ? '#ef4444' : '#f59e0b';
      const displayValue = formatFieldValue(field.value, field.key);

      html += `
        <div class="field-row" data-key="${field.key}" data-value="${escapeHtml(String(field.value || ''))}" style="
          padding: 6px 8px;
          border: 1px solid #e5e7eb;
          border-radius: 4px;
          margin-bottom: 4px;
          transition: all 0.15s;
          display: flex;
          align-items: center;
          gap: 8px;
        ">
          <span style="
            color: ${statusColor};
            font-size: 13px;
            width: 14px;
            flex-shrink: 0;
            text-align: center;
          ">${statusIcon}</span>
          <div style="flex: 1; min-width: 0;">
            <div style="
              font-size: 11px;
              font-weight: 500;
              color: #9ca3af;
              margin-bottom: 2px;
            ">${field.label}${field.required ? ' <span style="color: #ef4444;">*</span>' : ''}</div>
            <div style="
              font-size: 13px;
              color: ${field.value ? '#111827' : '#d1d5db'};
              word-break: break-word;
              line-height: 1.3;
            ">${displayValue}</div>
          </div>
          ${field.value ? `
            <button class="copy-field-btn" data-value="${escapeHtml(String(field.value))}" style="
              padding: 4px 8px;
              background: #f9fafb;
              border: 1px solid #e5e7eb;
              border-radius: 3px;
              font-size: 10px;
              color: #6b7280;
              cursor: pointer;
              transition: all 0.15s;
              white-space: nowrap;
              flex-shrink: 0;
            ">Copy</button>
          ` : ''}
        </div>
      `;
    });

    html += `</div>`;
  });

  // Show "no results" message if search returned nothing
  if (searchQuery && html === '') {
    html = `
      <div style="text-align: center; padding: 40px 16px; color: #9ca3af;">
        <div style="font-size: 40px; margin-bottom: 12px;">🔍</div>
        <p style="margin: 0; font-size: 13px;">No fields match "${escapeHtml(searchQuery)}"</p>
      </div>
    `;
  }

  content.innerHTML = html;

  // Update header with search results count
  const headerSubtitleElement = document.querySelector('#ghl-utils-field-inspector-panel p');
  if (headerSubtitleElement) {
    if (searchQuery && totalFilteredFields > 0) {
      headerSubtitleElement.textContent = `Found ${totalFilteredFields} field${totalFilteredFields !== 1 ? 's' : ''}`;
    } else if (!searchQuery) {
      const baseText = headerSubtitleElement.getAttribute('data-base-text');
      if (baseText) {
        headerSubtitleElement.textContent = baseText;
      }
    }
  }

  // Add hover effects and click handlers
  document.querySelectorAll('.field-row').forEach(row => {
    row.addEventListener('mouseenter', () => {
      row.style.background = '#f9fafb';
      row.style.borderColor = '#d1d5db';
      row.style.cursor = 'default';
    });
    row.addEventListener('mouseleave', () => {
      row.style.background = 'transparent';
      row.style.borderColor = '#e5e7eb';
    });
  });

  // Copy button handlers
  document.querySelectorAll('.copy-field-btn').forEach(btn => {
    btn.addEventListener('click', (e) => {
      e.stopPropagation();
      const value = btn.getAttribute('data-value');
      navigator.clipboard.writeText(value).then(() => {
        const originalText = btn.textContent;
        btn.textContent = '✓ Copied';
        btn.style.background = '#10b981';
        btn.style.color = 'white';
        btn.style.borderColor = '#10b981';

        setTimeout(() => {
          btn.textContent = originalText;
          btn.style.background = '#f3f4f6';
          btn.style.color = '#6b7280';
          btn.style.borderColor = '#d1d5db';
        }, 2000);
      });
    });

    btn.addEventListener('mouseenter', () => {
      btn.style.background = '#e5e7eb';
      btn.style.borderColor = '#9ca3af';
    });
    btn.addEventListener('mouseleave', () => {
      btn.style.background = '#f3f4f6';
      btn.style.borderColor = '#d1d5db';
    });
  });
}

// Calculate data quality score
function calculateDataQuality(contact) {
  let total = 0;
  let filled = 0;

  STANDARD_FIELDS.forEach(field => {
    total++;
    if (contact[field.key] && String(contact[field.key]).trim() !== '') {
      filled++;
    }
  });

  fieldInspectorState.customFields.forEach(customField => {
    total++;

    // Try multiple possible locations for custom field values
    let value = null;

    // Method 1: customField object (singular)
    if (contact.customField && typeof contact.customField === 'object') {
      value = contact.customField[customField.id];
    }

    // Method 2: customFields array
    if (!value && Array.isArray(contact.customFields)) {
      const customFieldData = contact.customFields.find(cf => cf.id === customField.id);
      value = customFieldData?.value;
    }

    // Method 3: Direct property on contact object
    if (!value && contact[customField.id]) {
      value = contact[customField.id];
    }

    if (value && String(value).trim() !== '') {
      filled++;
    }
  });

  const percentage = total > 0 ? Math.round((filled / total) * 100) : 0;

  return { total, filled, percentage };
}

// Get field status
function getFieldStatus(value, required) {
  if (!value || String(value).trim() === '') {
    return required ? 'missing' : 'empty';
  }

  // Could add more validation rules here
  // For example, email format validation, phone format, etc.

  return 'valid';
}

// Format field value for display
function formatFieldValue(value, key) {
  if (!value || String(value).trim() === '') {
    return '<em>Not set</em>';
  }

  // Format arrays (like tags)
  if (Array.isArray(value)) {
    return value.join(', ');
  }

  // Format dates
  if (key.includes('date') || key.includes('Date')) {
    try {
      const date = new Date(value);
      return date.toLocaleDateString();
    } catch {
      return value;
    }
  }

  return String(value);
}

// Escape HTML
function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Watch for contact page navigation
function watchForContactPageNavigation() {
  let lastUrl = location.href;

  new MutationObserver(() => {
    if (location.href !== lastUrl) {
      lastUrl = location.href;

      // Remove existing button
      const existingBtn = document.getElementById('ghl-utils-field-inspector-btn');
      if (existingBtn) existingBtn.remove();

      // Close inspector if open
      if (fieldInspectorState.isOpen) {
        closeFieldInspector();
      }

      // Add button if on contact page
      if (isContactDetailPage()) {
        setTimeout(() => {
          addInspectorButton();
        }, 500);
      }

      // Setup list view handlers if on contacts list
      if (isContactsListPage()) {
        setTimeout(() => {
          addListViewClickHandlers();
        }, 500);
      }
    }
  }).observe(document.body, {
    subtree: true,
    childList: true
  });
}

// Watch for contact data changes on current page
function watchForContactChanges() {
  // If inspector is open and contact data changes, refresh
  const observer = new MutationObserver(() => {
    if (fieldInspectorState.isOpen) {
      const newContactId = extractContactId();
      if (newContactId && newContactId !== fieldInspectorState.currentContactId) {
        fetchAndDisplayContactData();
      }
    }
  });

  observer.observe(document.body, {
    subtree: true,
    childList: true
  });
}

// Show toast helper
function showToast(message) {
  if (typeof window.ghlUtilsShowToast === 'function') {
    window.ghlUtilsShowToast(message);
  }
}

// Listen for settings updates
chrome.runtime.onMessage.addListener((request, _sender, sendResponse) => {
  if (request.action === 'updateFieldInspectorSetting') {
    fieldInspectorState.enabled = request.enabled;

    // Remove button if disabled
    if (!fieldInspectorState.enabled) {
      const btn = document.getElementById('ghl-utils-field-inspector-btn');
      if (btn) btn.remove();

      if (fieldInspectorState.isOpen) {
        closeFieldInspector();
      }
    } else if (isContactDetailPage()) {
      addInspectorButton();
    }

    sendResponse({ success: true });
  }
});

// Initialize when content script loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initFieldInspector);
} else {
  initFieldInspector();
}
