// Lightweight content script for custom domains
// Only activates on pages that have GHL Nuxt data

(async function() {
  // Skip if we're already on a gohighlevel.com domain
  if (window.location.hostname.includes('gohighlevel.com')) {
    return;
  }

  // Check if this is a GHL page by trying to detect Nuxt data
  let isGHLPage = false;
  let pageData = null;

  // Inject detection script (external file to bypass CSP)
  const detectScript = document.createElement('script');
  detectScript.src = chrome.runtime.getURL('detect-ghl.js');
  detectScript.onload = function() {
    this.remove();
  };
  (document.head || document.documentElement).appendChild(detectScript);

  // Wait for detection response
  const detectionResult = await new Promise((resolve) => {
    const timeout = setTimeout(() => resolve({ isGHL: false }), 1000);

    const messageHandler = (event) => {
      if (event.data && event.data.type === 'GHL_CUSTOM_DOMAIN_DETECTION') {
        clearTimeout(timeout);
        window.removeEventListener('message', messageHandler);
        resolve({ isGHL: event.data.isGHL, data: event.data.data });
      }
    };

    window.addEventListener('message', messageHandler);
  });

  // Exit if not a GHL page
  if (!detectionResult.isGHL) {
    return;
  }

  isGHLPage = true;
  pageData = detectionResult.data || {};

  // Store location ID if found
  let currentLocationId = pageData.locationId || pageData.location_id;
  let cachedPageInfo = {
    pageId: pageData.pageId || pageData.id || pageData.page_id,
    funnelId: pageData.funnelId || pageData.funnel_id,
    stepId: pageData.stepId || pageData.step_id,
    locationId: currentLocationId,
    websiteId: pageData.websiteId || pageData.website_id,
    type: 'custom-domain',
    url: window.location.href,
    title: document.title,
    isGHLPage: true
  };

  // Save location ID to storage
  if (currentLocationId) {
    chrome.storage.sync.set({ locationId: currentLocationId });
  }

  // Function to extract page info with async injection
  async function extractGHLPageInfoAsync(timeoutMs = 500) {
    return new Promise((resolve) => {
      // Inject script to get fresh Nuxt data
      const extractScript = document.createElement("script");
      extractScript.src = chrome.runtime.getURL("inject.js");
      extractScript.onload = function() {
        this.remove();
      };
      (document.head || document.documentElement).appendChild(extractScript);

      let resolved = false;
      const messageHandler = (event) => {
        if (event.data && event.data.type === "GHL_PAGE_DATA_RESPONSE" && !resolved) {
          resolved = true;
          window.removeEventListener("message", messageHandler);
          clearTimeout(timeout);

          const data = event.data.data;
          const info = {
            pageId: data.pageId || data.id || data.page_id || cachedPageInfo.pageId,
            funnelId: data.funnelId || data.funnel_id || cachedPageInfo.funnelId,
            stepId: data.stepId || data.step_id || cachedPageInfo.stepId,
            locationId: data.locationId || data.location_id || cachedPageInfo.locationId,
            websiteId: data.websiteId || data.website_id || cachedPageInfo.websiteId,
            type: 'custom-domain',
            url: window.location.href,
            title: document.title,
            isGHLPage: true
          };

          // Update cached info
          cachedPageInfo = info;

          // Update location ID
          if (info.locationId) {
            currentLocationId = info.locationId;
            chrome.storage.sync.set({ locationId: currentLocationId });
          }

          resolve(info);
        }
      };

      window.addEventListener("message", messageHandler);

      const timeout = setTimeout(() => {
        if (!resolved) {
          resolved = true;
          window.removeEventListener("message", messageHandler);
          resolve(cachedPageInfo);
        }
      }, timeoutMs);
    });
  }

  // Function to generate page builder URL
  function generateBuilderUrl(info) {
    if (!info.locationId) return null;
    const pageId = info.pageId;
    if (!pageId) return null;
    return `https://app.gohighlevel.com/location/${info.locationId}/page-builder/${pageId}`;
  }

  // Function to show toast notification
  function showToast(message) {
    const toast = document.createElement('div');
    toast.textContent = message;
    toast.style.cssText = `
      position: fixed;
      top: 20px;
      right: 20px;
      background: #333;
      color: white;
      padding: 12px 24px;
      border-radius: 4px;
      z-index: 999999;
      font-family: -apple-system, BlinkMacSystemFont, "Segoe UI", Roboto, sans-serif;
      font-size: 14px;
      box-shadow: 0 4px 6px rgba(0,0,0,0.3);
    `;
    document.body.appendChild(toast);
    setTimeout(() => toast.remove(), 3000);
  }

  // Keyboard shortcut handler for opening page builder
  document.addEventListener("keydown", async (e) => {
    if (e.ctrlKey && e.shiftKey && e.key === "B") {
      e.preventDefault();
      const info = await extractGHLPageInfoAsync();
      const url = generateBuilderUrl(info);
      if (url) {
        window.open(url, "_blank");
        showToast("Opening page builder...");
      } else {
        showToast("⚠️ No page builder URL found");
      }
    }
  });

  // Message listener for extension commands
  chrome.runtime.onMessage.addListener((request, _sender, sendResponse) => {
    if (request.action === 'extractPageInfo') {
      (async () => {
        const pageInfo = await extractGHLPageInfoAsync();
        sendResponse(pageInfo);
      })();
      return true;
    }

    if (request.action === 'openPageBuilder') {
      (async () => {
        const pageInfo = await extractGHLPageInfoAsync();
        const builderUrl = generateBuilderUrl(pageInfo);

        if (builderUrl) {
          window.open(builderUrl, "_blank");
          showToast("Opening page builder...");
          sendResponse({ success: true, url: builderUrl });
        } else {
          showToast("⚠️ Unable to generate builder URL");
          sendResponse({
            success: false,
            error: "Could not generate builder URL",
          });
        }
      })();
      return true;
    }

    if (request.action === 'getLocationId') {
      sendResponse({ locationId: currentLocationId });
      return true;
    }
  });

  // Notify background that we detected a GHL page
  chrome.runtime.sendMessage({
    action: 'updateBadge',
    isGHLPage: true
  }).catch(() => {
    // Ignore errors if background script isn't ready
  });
})();
