// Command palette overlay for keyboard shortcut access
let paletteOpen = false;
let paletteElement = null;
let currentLocationId = null;

// GHL ID Regex Patterns
// Pattern 1: UUID format with hyphens (e.g., bd2b504c-d805-4198-854c-880f849bc5a6, 93a5e3cb-0ecb-453e-b4e1-0191c5c123b0)
const GHL_UUID_REGEX = /[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}/gi;
// Pattern 2: Standard alphanumeric IDs (e.g., l2Y3WbrBsxoFLwZdBYfj, XmdhW9ddEJz5GohRescv, fYZ9jS07Fca7WXQgdvLj)
const GHL_ID_REGEX = /\b[A-Za-z0-9]{16,32}\b/g;
// Pattern 3: Mixed format IDs with hyphens (catch-all for various ID formats)
const GHL_MIXED_ID_REGEX = /\b[A-Za-z0-9]{8,}-[A-Za-z0-9]{4,}-[A-Za-z0-9]{4,}-[A-Za-z0-9]{4,}-[A-Za-z0-9]{8,}\b/gi;

// Combined function to test both patterns
function isGhlId(text) {
  return GHL_UUID_REGEX.test(text) || GHL_ID_REGEX.test(text) || GHL_MIXED_ID_REGEX.test(text);
}

function getAllGhlIds(text) {
  // Reset regex lastIndex to ensure proper matching
  GHL_UUID_REGEX.lastIndex = 0;
  GHL_ID_REGEX.lastIndex = 0;
  GHL_MIXED_ID_REGEX.lastIndex = 0;

  const uuidMatches = text.match(GHL_UUID_REGEX) || [];
  const idMatches = text.match(GHL_ID_REGEX) || [];
  const mixedMatches = text.match(GHL_MIXED_ID_REGEX) || [];

  // Remove duplicates and filter out common false positives
  const allMatches = [...new Set([...uuidMatches, ...idMatches, ...mixedMatches])];

  // Filter out IDs that are too generic (all same character, too short, etc.)
  return allMatches.filter(id => {
    // Skip if it's all digits (could be timestamps, phone numbers, etc.)
    if (/^\d+$/.test(id)) return false;
    // Skip if it's too short
    if (id.length < 16) return false;
    return true;
  });
}

// Initialize contact ID variable first (must be before function calls)
let currentContactId = null;

// Inject CSS animations for toasts and other UI elements
(function injectStyles() {
  if (document.getElementById('ghl-utils-styles')) return; // Already injected

  const style = document.createElement('style');
  style.id = 'ghl-utils-styles';
  style.textContent = `
    @keyframes slideInRight {
      from {
        transform: translateX(400px);
        opacity: 0;
      }
      to {
        transform: translateX(0);
        opacity: 1;
      }
    }

    @keyframes slideOutRight {
      from {
        transform: translateX(0);
        opacity: 1;
      }
      to {
        transform: translateX(400px);
        opacity: 0;
      }
    }
  `;
  document.head.appendChild(style);
})();

// Detect location ID from URL on page load
detectAndSaveLocationId();

// Detect contact ID from URL parameters
detectContactIdFromUrl();

// Add audit log button on contact pages
addAuditLogButton();

// Listen for command from extension
chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
  if (request.action === 'toggleCommandPalette') {
    toggleCommandPalette();
  } else if (request.action === 'getLocationId') {
    sendResponse({ locationId: currentLocationId });
  } else if (request.action === 'getContactId') {
    // Return contact ID from current page
    const url = window.location.href;
    const contactMatch = url.match(/\/contacts\/detail\/([A-Za-z0-9_-]{16,32})/);

    if (contactMatch) {
      sendResponse({ contactId: contactMatch[1] });
    } else if (currentContactId) {
      sendResponse({ contactId: currentContactId });
    } else {
      sendResponse({ contactId: null });
    }
  }
  return true; // Keep message channel open for async response
});

// Auto-detect location ID from current URL
function detectAndSaveLocationId() {
  try {
    const url = window.location.href;
    // More flexible regex to catch various location ID formats
    const locationMatch = url.match(/\/location\/([A-Za-z0-9_-]{16,32})/) ||
                          url.match(/locationId=([A-Za-z0-9_-]{16,32})/) ||
                          url.match(/\/v2\/location\/([A-Za-z0-9_-]{16,32})/);
    const isAuditPage = url.includes('/settings/audit/logs');

    if (locationMatch && locationMatch[1]) {
      currentLocationId = locationMatch[1];

      // Save to storage for use by popup
      chrome.storage.sync.set({ locationId: currentLocationId }, () => {
        // After detecting location, make IDs clickable
        // Audit pages need more aggressive retries due to dynamic loading
        if (isAuditPage) {
          setTimeout(makeIdsClickable, 2000);
          setTimeout(makeIdsClickable, 4000);
          setTimeout(makeIdsClickable, 6000);
          setTimeout(makeIdsClickable, 8000);
        } else {
          setTimeout(makeIdsClickable, 2000);
          setTimeout(makeIdsClickable, 4000);
        }
      });
    } else {
      // Still try to get it from storage
      chrome.storage.sync.get(['locationId'], (result) => {
        if (result.locationId) {
          currentLocationId = result.locationId;

          // Audit pages need more aggressive retries
          if (isAuditPage) {
            setTimeout(makeIdsClickable, 2000);
            setTimeout(makeIdsClickable, 4000);
            setTimeout(makeIdsClickable, 6000);
            setTimeout(makeIdsClickable, 8000);
          } else {
            setTimeout(makeIdsClickable, 1000);
          }
        }
      });
    }
  } catch (error) {
    console.error('[GHL Utils] Error detecting location ID:', error);
  }
}

// Make GHL IDs clickable on the page
function makeIdsClickable() {
  if (!currentLocationId) {
    return;
  }

  let totalLinksCreated = 0;

  // Target specific containers - prioritize audit log table cells
  const targetSelectors = [
    '.hr-ellipsis span', // Direct spans inside hr-ellipsis (where IDs are)
    '.hr-data-table__body-cell', // Audit log specific
    'td[data-v-932be778]', // Audit log table cells
    '.hr-ellipsis', // Text containers with IDs
    'table td',
    '[class*="table"]',
    '[class*="list"]',
    '[class*="grid"]',
    'pre',
    'code'
  ];

  targetSelectors.forEach(selector => {
    const containers = document.querySelectorAll(selector);

    containers.forEach((container) => {
      if (container.hasAttribute('data-ghl-processed')) {
        // Already processed, skip
        return;
      }

      container.setAttribute('data-ghl-processed', 'true');

      const linkCount = processNodeForIds(container);
      totalLinksCreated += linkCount;
    });
  });

}

// Process a node and its text content to linkify IDs
function processNodeForIds(node) {
  let linksCreated = 0;

  if (node.nodeType === Node.TEXT_NODE) {
    const text = node.textContent.trim();
    if (!text) return 0;

    // Get all matching IDs (both UUIDs and standard IDs)
    const matches = getAllGhlIds(text);

    if (matches && matches.length > 0) {
      const fragment = document.createDocumentFragment();
      let remainingText = text;
      let processedAny = false;

      matches.forEach(id => {
        const index = remainingText.indexOf(id);
        if (index === -1) return;

        // Add text before the ID
        if (index > 0) {
          fragment.appendChild(document.createTextNode(remainingText.substring(0, index)));
        }

        // Create clickable link for ID - pass parent element for context detection
        const link = createIdLink(id, node.parentElement);
        fragment.appendChild(link);
        linksCreated++;

        remainingText = remainingText.substring(index + id.length);
        processedAny = true;
      });

      // Add remaining text
      if (remainingText) {
        fragment.appendChild(document.createTextNode(remainingText));
      }

      if (processedAny && node.parentNode) {
        node.parentNode.replaceChild(fragment, node);
      }
    }
  } else if (node.nodeType === Node.ELEMENT_NODE) {
    // Skip if already a link or script/style or already processed
    if (node.tagName === 'A' ||
        node.tagName === 'SCRIPT' ||
        node.tagName === 'STYLE' ||
        node.classList?.contains('ghl-utils-id-link')) {
      return 0;
    }

    // Process child nodes
    Array.from(node.childNodes).forEach(child => {
      linksCreated += processNodeForIds(child);
    });
  }

  return linksCreated;
}

// Detect resource type from context
function detectResourceType(element) {
  // Look at the parent elements and text content to detect type
  let current = element;
  let attempts = 0;

  while (current && attempts < 5) {
    const text = current.textContent || '';
    const lowerText = text.toLowerCase();

    // Check for resource type indicators
    if (lowerText.includes('workflow')) return 'workflow';
    if (lowerText.includes('form') && !lowerText.includes('survey')) return 'form';
    if (lowerText.includes('survey')) return 'survey';
    if (lowerText.includes('funnel') || lowerText.includes('page')) return 'page';
    if (lowerText.includes('calendar')) return 'calendar';
    if (lowerText.includes('contact')) return 'contact';
    if (lowerText.includes('integration')) return 'integration';

    current = current.parentElement;
    attempts++;
  }

  // Default to workflow if can't determine
  return 'workflow';
}

// Create a clickable link for an ID
function createIdLink(id, contextElement = null) {
  const link = document.createElement('a');
  link.href = '#';
  link.textContent = id;
  link.className = 'ghl-utils-id-link';
  link.title = 'Click to open this resource (Right-click for options)';

  // Detect resource type from context
  const resourceType = contextElement ? detectResourceType(contextElement) : 'workflow';
  link.setAttribute('data-resource-type', resourceType);

  link.style.cssText = `
    color: #D84315;
    text-decoration: underline;
    cursor: pointer;
    font-weight: 500;
  `;

  link.addEventListener('click', (e) => {
    e.preventDefault();
    e.stopPropagation();

    // Open using detected resource type
    const type = link.getAttribute('data-resource-type') || 'workflow';
    openResourceById(id, type);
  });

  link.addEventListener('contextmenu', (e) => {
    e.preventDefault();
    showIdContextMenu(id, e.clientX, e.clientY);
  });

  return link;
}

// Open resource by ID and type
function openResourceById(id, type) {
  const message = {
    action: 'navigate',
    type: type,
    id: id
  };

  // Include contactId if available and it's a workflow
  if (currentContactId && type === 'workflow') {
    message.contactId = currentContactId;
  }

  chrome.runtime.sendMessage(message);

  // Show toast with resource type
  const typeLabels = {
    workflow: '⚙️ Workflow',
    page: '📄 Page/Funnel',
    form: '📝 Form',
    survey: '📊 Survey',
    contact: '👤 Contact',
    calendar: '📅 Calendar',
    integration: '🔗 Integration'
  };

  const label = typeLabels[type] || '📦 Resource';

  // Add contact filter indication for workflows
  if (type === 'workflow' && currentContactId) {
    showToast(`${label} (filtered by contact)...`);
  } else {
    showToast(`Opening ${label}...`);
  }
}

// Open command palette with pre-filled ID
function openCommandPaletteWithId(id) {
  // If there's a contactId in the URL, open workflow with contactId parameter
  if (currentContactId) {
    chrome.runtime.sendMessage({
      action: 'navigate',
      type: 'workflow',
      id: id,
      contactId: currentContactId
    });
    showToast(`⚙️ Opening workflow with contact filter...`);
    return;
  }

  // Otherwise, open command palette with workflow command
  openCommandPalette();

  setTimeout(() => {
    const input = document.getElementById('ghl-search-input');
    if (input) {
      input.value = `/w ${id}`;
      input.focus();
    }
  }, 150);
}

// Show context menu for ID
function showIdContextMenu(id, x, y) {
  // Remove existing menu if any
  const existingMenu = document.getElementById('ghl-id-context-menu');
  if (existingMenu) existingMenu.remove();

  const menu = document.createElement('div');
  menu.id = 'ghl-id-context-menu';
  menu.style.cssText = `
    position: fixed;
    left: ${x}px;
    top: ${y}px;
    background: white;
    border: 2px solid #FFB74D;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
    z-index: 9999999;
    padding: 8px 0;
    min-width: 200px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  const menuItems = [
    { icon: '⚙️', label: 'Open as Workflow', action: () => openAsType(id, 'workflow') },
    { icon: '📄', label: 'Open as Page/Funnel', action: () => openAsType(id, 'page') },
    { icon: '📝', label: 'Open as Form', action: () => openAsType(id, 'form') },
    { icon: '📊', label: 'Open as Survey', action: () => openAsType(id, 'survey') },
    { icon: '👤', label: 'Open as Contact', action: () => openAsType(id, 'contact') },
    { icon: '📋', label: 'Open Audit Logs', action: () => openAsType(id, 'audit') },
    { type: 'divider' },
    { icon: '📋', label: 'Copy ID', action: () => copyToClipboard(id) }
  ];

  menuItems.forEach(item => {
    if (item.type === 'divider') {
      const divider = document.createElement('div');
      divider.style.cssText = 'height: 1px; background: #FFE0B2; margin: 4px 0;';
      menu.appendChild(divider);
    } else {
      const menuItem = document.createElement('div');
      menuItem.style.cssText = `
        padding: 10px 16px;
        cursor: pointer;
        display: flex;
        align-items: center;
        gap: 10px;
        transition: background 0.2s;
      `;
      menuItem.innerHTML = `<span style="font-size: 16px;">${item.icon}</span><span>${item.label}</span>`;

      menuItem.addEventListener('mouseenter', () => {
        menuItem.style.background = '#FFF3E0';
      });

      menuItem.addEventListener('mouseleave', () => {
        menuItem.style.background = 'transparent';
      });

      menuItem.addEventListener('click', () => {
        item.action();
        menu.remove();
      });

      menu.appendChild(menuItem);
    }
  });

  document.body.appendChild(menu);

  // Close menu when clicking outside
  const closeMenu = (e) => {
    if (!menu.contains(e.target)) {
      menu.remove();
      document.removeEventListener('click', closeMenu);
    }
  };
  setTimeout(() => document.addEventListener('click', closeMenu), 100);
}

// Open ID as specific type
function openAsType(id, type) {
  const message = {
    action: 'navigate',
    type: type,
    id: id
  };

  // Include contactId if available
  if (currentContactId) {
    message.contactId = currentContactId;
  }

  chrome.runtime.sendMessage(message);
}

// Copy to clipboard
function copyToClipboard(text) {
  navigator.clipboard.writeText(text).then(() => {
    showToast('ID copied to clipboard!');
  });
}

// Show toast notification
function showToast(message) {
  const toast = document.createElement('div');
  toast.style.cssText = `
    position: fixed;
    bottom: 20px;
    right: 20px;
    background: linear-gradient(135deg, #4CAF50 0%, #66BB6A 100%);
    color: white;
    padding: 12px 20px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
    z-index: 99999999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    font-size: 14px;
    transform: translateX(400px);
    opacity: 0;
    transition: all 0.3s ease-out;
  `;
  toast.textContent = message;

  document.body.appendChild(toast);

  // Trigger slide-in animation
  requestAnimationFrame(() => {
    requestAnimationFrame(() => {
      toast.style.transform = 'translateX(0)';
      toast.style.opacity = '1';
    });
  });

  // Auto-hide after 2 seconds
  setTimeout(() => {
    toast.style.transform = 'translateX(400px)';
    toast.style.opacity = '0';
    setTimeout(() => {
      if (toast.parentNode) {
        toast.remove();
      }
    }, 300);
  }, 2000);
}

// Export showToast for use by other modules (like bulk-contact-editor)
window.ghlUtilsShowToast = showToast;

// Observe DOM changes to make new IDs clickable
let observerTimeout;
const observer = new MutationObserver((mutations) => {
  if (currentLocationId) {
    // Debounce to avoid excessive processing
    clearTimeout(observerTimeout);
    observerTimeout = setTimeout(() => {
      // Remove processed markers from new or changed elements
      mutations.forEach(mutation => {
        mutation.addedNodes.forEach(node => {
          if (node.nodeType === Node.ELEMENT_NODE) {
            // Remove data-ghl-processed from added nodes to allow reprocessing
            node.querySelectorAll && node.querySelectorAll('[data-ghl-processed]').forEach(el => {
              el.removeAttribute('data-ghl-processed');
            });
            if (node.hasAttribute && node.hasAttribute('data-ghl-processed')) {
              node.removeAttribute('data-ghl-processed');
            }
          }
        });
      });

      makeIdsClickable();
    }, 500);
  }
});

// Start observing after page load
if (document.body) {
  observer.observe(document.body, {
    childList: true,
    subtree: true
  });
}

// Also run when page becomes visible (tab switching)
document.addEventListener('visibilitychange', () => {
  if (!document.hidden && currentLocationId) {
    setTimeout(makeIdsClickable, 500);
    setTimeout(addAuditLogButton, 500);
  }
});

// Detect contact ID from URL parameters
function detectContactIdFromUrl() {
  try {
    const urlParams = new URLSearchParams(window.location.search);
    const url = window.location.href;

    // First try to get contactId parameter
    let contactId = urlParams.get('contactId');

    // If on audit page, use the 'id' or 'documentId' parameter as contactId
    if (!contactId && url.includes('/settings/audit/logs')) {
      contactId = urlParams.get('id') || urlParams.get('documentId');
    }

    // Also check contact detail pages
    if (!contactId) {
      const contactMatch = url.match(/\/contacts\/detail\/([A-Za-z0-9_-]{16,32})/);
      if (contactMatch) {
        contactId = contactMatch[1];
      }
    }

    if (contactId) {
      currentContactId = contactId;

      // Show indicator on audit logs page
      if (url.includes('/settings/audit/logs')) {
        showContactFilterIndicator(contactId);
      }
    } else {
      currentContactId = null; // Clear if not found
      removeContactFilterIndicator();
    }
  } catch (error) {
    console.error('[GHL Utils] Error detecting contactId:', error);
  }
}

// Show contact filter indicator on audit logs page
function showContactFilterIndicator(contactId) {
  // Remove existing indicator if any
  removeContactFilterIndicator();

  const indicator = document.createElement('div');
  indicator.id = 'ghl-utils-contact-filter-indicator';
  indicator.style.cssText = `
    position: fixed;
    bottom: 80px;
    right: 20px;
    background: linear-gradient(135deg, #2196F3 0%, #42A5F5 100%);
    color: white;
    padding: 12px 16px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
    z-index: 99999999;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    font-size: 13px;
    display: flex;
    align-items: center;
    gap: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
  `;

  indicator.innerHTML = `
    <span style="font-size: 16px;">🔗</span>
    <div>
      <div style="font-weight: 600;">Workflow Filter Active</div>
      <div style="font-size: 11px; opacity: 0.9;">Workflows will open filtered by contact</div>
    </div>
  `;

  indicator.addEventListener('mouseenter', () => {
    indicator.style.transform = 'translateY(-4px)';
    indicator.style.boxShadow = '0 6px 16px rgba(0, 0, 0, 0.4)';
  });

  indicator.addEventListener('mouseleave', () => {
    indicator.style.transform = 'translateY(0)';
    indicator.style.boxShadow = '0 4px 12px rgba(0, 0, 0, 0.3)';
  });

  indicator.addEventListener('click', () => {
    showToast(`Workflows will open at /status?contactId=${contactId.substring(0, 8)}...`);
  });

  document.body.appendChild(indicator);

  // Auto-hide after 5 seconds
  setTimeout(() => {
    if (indicator.parentNode) {
      indicator.style.opacity = '0.7';
      indicator.style.transform = 'scale(0.9)';
    }
  }, 5000);
}

function removeContactFilterIndicator() {
  const existing = document.getElementById('ghl-utils-contact-filter-indicator');
  if (existing) {
    existing.remove();
  }
}

// Watch for URL changes in SPA (GHL uses client-side routing)
let lastUrl = window.location.href;

function handleUrlChange() {
  const currentUrl = window.location.href;
  if (currentUrl !== lastUrl) {
    lastUrl = currentUrl;

    // Re-detect contact ID on URL change
    detectContactIdFromUrl();

    // Re-detect location ID on URL change
    detectAndSaveLocationId();

    // Re-make IDs clickable after a short delay (for new content to load)
    setTimeout(makeIdsClickable, 1000);
    setTimeout(addAuditLogButton, 1000);
  }
}

// Method 1: MutationObserver on title/body
const urlObserver = new MutationObserver(handleUrlChange);
urlObserver.observe(document.querySelector('head > title') || document.body, {
  subtree: true,
  characterData: true,
  childList: true
});

// Method 2: Listen for popstate (back/forward navigation)
window.addEventListener('popstate', handleUrlChange);

// Method 3: Intercept pushState and replaceState for SPA navigation
const originalPushState = history.pushState;
const originalReplaceState = history.replaceState;

history.pushState = function() {
  originalPushState.apply(this, arguments);
  handleUrlChange();
};

history.replaceState = function() {
  originalReplaceState.apply(this, arguments);
  handleUrlChange();
};

// Add audit log button on contact detail pages only
function addAuditLogButton() {
  // Check if we're on a contact detail page
  const url = window.location.href;
  const contactMatch = url.match(/\/contacts\/detail\/([A-Za-z0-9_-]{16,32})/);

  // Only show button on contact detail pages
  if (!contactMatch) {
    return;
  }

  const contactId = contactMatch[1];

  // Check if button already exists
  if (document.querySelector('#ghl-utils-audit-btn')) {
    return;
  }

  // Wait for page to load and find a good spot to inject the button
  setTimeout(() => {
    // Try multiple possible locations for the button
    const possibleContainers = [
      '.contact-detail-header',
      '.contact-header',
      '[class*="contact"][class*="header"]',
      'header',
      '.page-header',
      '.toolbar'
    ];

    let targetContainer = null;
    for (const selector of possibleContainers) {
      targetContainer = document.querySelector(selector);
      if (targetContainer) break;
    }

    // If no specific container found, create a floating button
    if (!targetContainer) {
      createFloatingAuditButton(contactId);
    } else {
      createInlineAuditButton(contactId, targetContainer);
    }
  }, 1000);
}

function createFloatingAuditButton(contactId, buttonText = '📋 View Audit Logs') {
  const button = document.createElement('button');
  button.id = 'ghl-utils-audit-btn';
  button.innerHTML = buttonText;
  button.style.cssText = `
    position: fixed;
    top: 80px;
    right: 20px;
    background: linear-gradient(135deg, #D84315 0%, #FF6F00 100%);
    color: white;
    border: none;
    padding: 12px 20px;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 600;
    cursor: pointer;
    z-index: 999999;
    box-shadow: 0 4px 12px rgba(216, 67, 21, 0.3);
    transition: all 0.2s;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  button.addEventListener('mouseenter', () => {
    button.style.transform = 'translateY(-2px)';
    button.style.boxShadow = '0 6px 16px rgba(216, 67, 21, 0.4)';
  });

  button.addEventListener('mouseleave', () => {
    button.style.transform = 'translateY(0)';
    button.style.boxShadow = '0 4px 12px rgba(216, 67, 21, 0.3)';
  });

  button.addEventListener('click', () => {
    openAuditLogsForContact(contactId);
  });

  document.body.appendChild(button);
}

function createInlineAuditButton(contactId, container, buttonText = '📋 Audit Logs') {
  const button = document.createElement('button');
  button.id = 'ghl-utils-audit-btn';
  button.innerHTML = buttonText;
  button.style.cssText = `
    background: linear-gradient(135deg, #D84315 0%, #FF6F00 100%);
    color: white;
    border: none;
    padding: 10px 18px;
    border-radius: 6px;
    font-size: 13px;
    font-weight: 600;
    cursor: pointer;
    margin-left: 10px;
    box-shadow: 0 2px 6px rgba(216, 67, 21, 0.3);
    transition: all 0.2s;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  button.addEventListener('mouseenter', () => {
    button.style.transform = 'translateY(-2px)';
    button.style.boxShadow = '0 4px 10px rgba(216, 67, 21, 0.4)';
  });

  button.addEventListener('mouseleave', () => {
    button.style.transform = 'translateY(0)';
    button.style.boxShadow = '0 2px 6px rgba(216, 67, 21, 0.3)';
  });

  button.addEventListener('click', () => {
    openAuditLogsForContact(contactId);
  });

  container.appendChild(button);
}

function openAuditLogsForContact(contactId) {
  if (!currentLocationId) {
    showToast('⚠️ Location ID not detected!');
    return;
  }

  const auditUrl = `https://app.gohighlevel.com/v2/location/${currentLocationId}/settings/audit/logs?id=${contactId}`;

  // Open in new tab next to current tab
  chrome.runtime.sendMessage({
    action: 'openTabNextToCurrent',
    url: auditUrl
  });

  showToast('✅ Opening audit logs...');
}

function createCommandPalette() {
  const overlay = document.createElement('div');
  overlay.id = 'ghl-utils-overlay';
  overlay.innerHTML = `
    <div class="ghl-utils-modal">
      <div class="ghl-utils-header">
        <h2>GHL Utils</h2>
        <button class="ghl-utils-close" id="ghl-close-btn">&times;</button>
      </div>
      <div class="ghl-utils-search">
        <input
          type="text"
          id="ghl-search-input"
          placeholder="Type /workflow, /page, /form, /survey or paste ID..."
          autocomplete="off"
        >
      </div>
      <div id="ghl-suggestions" class="ghl-suggestions"></div>
      <div class="ghl-utils-help">
        <div class="ghl-help-item"><code>/workflow</code> or <code>/w</code> - Open workflow</div>
        <div class="ghl-help-item"><code>/page</code> or <code>/p</code> - Open page/funnel</div>
        <div class="ghl-help-item"><code>/form</code> or <code>/f</code> - Open form</div>
        <div class="ghl-help-item"><code>/survey</code> or <code>/s</code> - Open survey</div>
        <div class="ghl-help-item"><code>/contact</code> or <code>/c</code> - Open contact</div>
        <div class="ghl-help-item"><code>/audit</code> or <code>/a</code> - Open audit logs (auto-detects)</div>
        <div class="ghl-help-item"><code>/builder</code> or <code>/b</code> - Open current page in builder</div>
      </div>
    </div>
  `;

  document.body.appendChild(overlay);
  return overlay;
}

function toggleCommandPalette() {
  if (paletteOpen) {
    closeCommandPalette();
  } else {
    openCommandPalette();
  }
}

function openCommandPalette() {
  if (!paletteElement) {
    paletteElement = createCommandPalette();
    setupPaletteHandlers();
  }

  paletteElement.style.display = 'flex';
  paletteOpen = true;

  // Focus the input
  setTimeout(() => {
    const input = document.getElementById('ghl-search-input');
    if (input) input.focus();
  }, 100);
}

function closeCommandPalette() {
  if (paletteElement) {
    paletteElement.style.display = 'none';
    const input = document.getElementById('ghl-search-input');
    if (input) input.value = '';
    const suggestions = document.getElementById('ghl-suggestions');
    if (suggestions) suggestions.innerHTML = '';
  }
  paletteOpen = false;
}

function setupPaletteHandlers() {
  const overlay = paletteElement;
  const input = document.getElementById('ghl-search-input');
  const closeBtn = document.getElementById('ghl-close-btn');
  const suggestions = document.getElementById('ghl-suggestions');

  // Close on overlay click
  overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
      closeCommandPalette();
    }
  });

  // Close button
  closeBtn.addEventListener('click', closeCommandPalette);

  // ESC key to close
  input.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') {
      closeCommandPalette();
    } else if (e.key === 'Enter') {
      handlePaletteNavigation(input.value);
    }
  });

  // Input suggestions
  input.addEventListener('input', (e) => {
    const value = e.target.value.trim();
    if (value.startsWith('/')) {
      showPaletteSuggestions(value, suggestions, input);
    } else {
      suggestions.innerHTML = '';
    }
  });
}

const COMMANDS = [
  { command: '/workflow', aliases: ['/w'], icon: '⚙️', description: 'Open workflow by ID', type: 'workflow' },
  { command: '/page', aliases: ['/p'], icon: '📄', description: 'Open funnel/page by ID', type: 'page' },
  { command: '/form', aliases: ['/f'], icon: '📝', description: 'Open form by ID', type: 'form' },
  { command: '/survey', aliases: ['/s'], icon: '📊', description: 'Open survey by ID', type: 'survey' },
  { command: '/contact', aliases: ['/c'], icon: '👤', description: 'Open contact by ID', type: 'contact' },
  { command: '/audit', aliases: ['/a'], icon: '📋', description: 'Open audit logs by ID', type: 'audit' },
  { command: '/builder', aliases: ['/b', '/pb'], icon: '🏗️', description: 'Open current page in builder', type: 'builder' }
];

function showPaletteSuggestions(input, suggestionsDiv, inputElement) {
  const query = input.toLowerCase();
  const queryCmd = query.split(' ')[0];
  const matches = COMMANDS.filter(cmd =>
    cmd.command.startsWith(queryCmd) ||
    (cmd.aliases && cmd.aliases.some(alias => alias.startsWith(queryCmd)))
  );

  if (matches.length === 0) {
    suggestionsDiv.innerHTML = '';
    return;
  }

  suggestionsDiv.innerHTML = matches.map(cmd => `
    <div class="ghl-suggestion-item" data-command="${cmd.command}">
      <span class="ghl-suggestion-icon">${cmd.icon}</span>
      <div class="ghl-suggestion-text">
        <div class="ghl-suggestion-command">${cmd.command}${cmd.aliases ? ` (${cmd.aliases.join(', ')})` : ''}</div>
        <div class="ghl-suggestion-desc">${cmd.description}</div>
      </div>
    </div>
  `).join('');

  // Add click handlers
  document.querySelectorAll('.ghl-suggestion-item').forEach(item => {
    item.addEventListener('click', () => {
      inputElement.value = item.dataset.command + ' ';
      inputElement.focus();
      suggestionsDiv.innerHTML = '';
    });
  });
}

function parseInput(input) {
  const trimmed = input.trim();

  if (trimmed.startsWith('/')) {
    const parts = trimmed.split(/\s+/);
    const command = parts[0];
    const id = parts.slice(1).join('').trim();

    // Check for exact command or alias match
    const cmdDef = COMMANDS.find(c =>
      c.command === command ||
      (c.aliases && c.aliases.includes(command))
    );

    if (cmdDef) {
      // Special case: /builder or /b or /pb without ID (open current page)
      if (cmdDef.type === 'builder') {
        return { type: 'builder' };
      }

      // Special case: /a or /audit without ID on contact page
      if (cmdDef.type === 'audit' && !id) {
        // Try to get contact ID from current page URL
        const url = window.location.href;
        const contactMatch = url.match(/\/contacts\/detail\/([A-Za-z0-9_-]{16,32})/);

        if (contactMatch) {
          return { type: 'audit', id: contactMatch[1] };
        } else if (currentContactId) {
          return { type: 'audit', id: currentContactId };
        }

        // No contact ID found
        return null;
      }

      // Normal case: command with ID
      if (id) {
        return { type: cmdDef.type, id };
      }
    }
  }

  if (trimmed.length > 0) {
    return { type: 'workflow', id: trimmed };
  }

  return null;
}

async function handlePaletteNavigation(input) {
  const parsed = parseInput(input);

  if (!parsed) {
    return;
  }

  // Special case: builder command opens current page in builder
  if (parsed.type === 'builder') {
    // Use cached info if available and recent, otherwise extract fresh
    const pageInfo = cachedPageInfo || await extractGHLPageInfoAsync();
    cachedPageInfo = pageInfo; // Update cache
    const builderUrl = generateBuilderUrl(pageInfo);

    if (builderUrl) {
      window.open(builderUrl, "_blank");
      showToast("Opening page builder...");
    } else {
      showToast("⚠️ No page builder URL found");
    }

    closeCommandPalette();
    return;
  }

  // Send message to background script to open URL
  chrome.runtime.sendMessage({
    action: 'navigate',
    type: parsed.type,
    id: parsed.id
  });

  closeCommandPalette();
}

// ============================================================================
// GHL PAGE OPENER INTEGRATION
// ============================================================================

// Store extracted page data
let cachedPageInfo = null;

// Extract all possible GHL page information
function extractGHLPageInfo() {
  const info = {
    pageId: null,
    funnelId: null,
    stepId: null,
    locationId: currentLocationId, // Use existing location ID detection
    websiteId: null,
    type: null,
    url: window.location.href,
    title: document.title,
    isGHLPage: false,
  };

  // Parse URL patterns (secondary detection)
  const patterns = {
    pageBuilder: /location\/([^\/]+)\/page-builder\/([^\/\?]+)/,
    funnelEditor: /location\/([^\/]+)\/funnels\/([^\/]+)\/editor\/([^\/\?]+)/,
    funnelPreview: /location\/([^\/]+)\/funnels\/([^\/]+)\/preview\/([^\/\?]+)/,
    websiteEditor: /location\/([^\/]+)\/websites\/([^\/]+)\/editor\/([^\/\?]+)/,
    v2Preview: /\/v2\/preview\/([^\/]+)\/([^\/\?]+)/,
    publicPage: /\/v2\/([^\/\?]+)/,
  };

  for (const [type, pattern] of Object.entries(patterns)) {
    const match = info.url.match(pattern);
    if (match) {
      info.isGHLPage = true;
      switch (type) {
        case "pageBuilder":
          info.type = "page-builder";
          info.locationId = match[1];
          info.pageId = match[2];
          break;
        case "funnelEditor":
        case "funnelPreview":
          info.type = "funnel";
          info.locationId = match[1];
          info.funnelId = match[2];
          info.stepId = match[3];
          info.pageId = match[3];
          break;
        case "websiteEditor":
          info.type = "website";
          info.locationId = match[1];
          info.websiteId = match[2];
          info.pageId = match[3];
          break;
        case "v2Preview":
          info.type = "preview";
          info.funnelId = match[1];
          info.stepId = match[2];
          info.pageId = match[2];
          break;
      }
      if (info.type) break;
    }
  }

  return info;
}

// Async version that waits for injected script response
function extractGHLPageInfoAsync(timeoutMs = 500) {
  return new Promise((resolve) => {
    const info = extractGHLPageInfo(); // Get URL-parsed data first

    // Inject script to bypass CSP restrictions and get Nuxt data
    const extractScript = document.createElement("script");
    extractScript.src = chrome.runtime.getURL("inject.js");
    extractScript.onload = function() {
      this.remove();
    };
    (document.head || document.documentElement).appendChild(extractScript);

    // Set up one-time message listener for this extraction
    let resolved = false;
    const messageHandler = (event) => {
      if (event.data && event.data.type === "GHL_PAGE_DATA_RESPONSE" && !resolved) {
        resolved = true;
        window.removeEventListener("message", messageHandler);
        clearTimeout(timeout);

        const data = event.data.data;
        const hasGHLData = event.data.hasGHLData;

        // Merge Nuxt data with URL-parsed data (Nuxt takes precedence)
        if (hasGHLData) {
          info.isGHLPage = true;
        }

        info.pageId = data.pageId || data.id || data.page_id || info.pageId;
        info.funnelId = data.funnelId || data.funnel_id || info.funnelId;
        info.stepId = data.stepId || data.step_id || info.stepId;
        info.locationId = data.locationId || data.location_id || info.locationId;
        info.websiteId = data.websiteId || data.website_id || info.websiteId;

        resolve(info);
      }
    };

    window.addEventListener("message", messageHandler);

    // Timeout fallback - return URL-parsed data if injection doesn't respond
    const timeout = setTimeout(() => {
      if (!resolved) {
        resolved = true;
        window.removeEventListener("message", messageHandler);
        resolve(info);
      }
    }, timeoutMs);
  });
}

// Generate page builder URL
function generateBuilderUrl(info) {
  if (!info.locationId) return null;

  const pageId = info.pageId;
  if (!pageId) return null;

  return `https://app.gohighlevel.com/location/${info.locationId}/page-builder/${pageId}`;
}

// Notify background about GHL page status
function notifyBackgroundAboutGHLPage(isGHLPage) {
  chrome.runtime.sendMessage({
    action: 'updateBadge',
    isGHLPage: isGHLPage
  }).catch(() => {
    // Ignore errors if background script isn't ready
  });
}

// Listen for data from injected script
window.addEventListener("message", (event) => {
  if (event.data && event.data.type === "GHL_PAGE_DATA_RESPONSE") {
    const data = event.data.data;
    const hasGHLData = event.data.hasGHLData;

    if (data) {
      // Create cachedPageInfo if it doesn't exist
      if (!cachedPageInfo) {
        cachedPageInfo = extractGHLPageInfo();
      }

      // Mark as GHL page if we got GHL data from Nuxt
      if (hasGHLData) {
        cachedPageInfo.isGHLPage = true;
        notifyBackgroundAboutGHLPage(true);
      }

      // Nuxt data takes precedence over URL parsing
      cachedPageInfo.pageId = data.pageId || data.id || data.page_id || cachedPageInfo.pageId;
      cachedPageInfo.funnelId = data.funnelId || data.funnel_id || cachedPageInfo.funnelId;
      cachedPageInfo.stepId = data.stepId || data.step_id || cachedPageInfo.stepId;
      cachedPageInfo.locationId = data.locationId || data.location_id || cachedPageInfo.locationId;
      cachedPageInfo.websiteId = data.websiteId || data.website_id || cachedPageInfo.websiteId;

      // Update global currentLocationId if we got it from Nuxt data
      if (cachedPageInfo.locationId) {
        currentLocationId = cachedPageInfo.locationId;

        // Save to storage for use by popup
        chrome.storage.sync.set({ locationId: currentLocationId });
      }
    }
  }
});

// Auto-detect GHL page on load and update badge
async function autoDetectAndUpdateBadge() {
  // First get URL-parsed data immediately
  cachedPageInfo = extractGHLPageInfo();

  if (cachedPageInfo.isGHLPage) {
    notifyBackgroundAboutGHLPage(true);
  }

  // Then get Nuxt data asynchronously
  const pageInfo = await extractGHLPageInfoAsync();
  cachedPageInfo = pageInfo;

  if (pageInfo.isGHLPage) {
    notifyBackgroundAboutGHLPage(true);
  }

  // Update global currentLocationId if we got it from page detection
  if (pageInfo.locationId) {
    currentLocationId = pageInfo.locationId;

    // Save to storage for use by popup
    chrome.storage.sync.set({ locationId: currentLocationId });
  }
}

// Run auto-detection
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', autoDetectAndUpdateBadge);
} else {
  autoDetectAndUpdateBadge();
}

// Keyboard shortcut handler for opening page builder
document.addEventListener("keydown", async (e) => {
  if (e.ctrlKey && e.shiftKey && e.key === "B") {
    e.preventDefault();
    const info = cachedPageInfo || await extractGHLPageInfoAsync();
    cachedPageInfo = info; // Update cache
    const url = generateBuilderUrl(info);
    if (url) {
      window.open(url, "_blank");
      showToast("Opening page builder...");
    } else {
      showToast("⚠️ No page builder URL found");
    }
  }
});

// Enhanced message listener for page opener actions
chrome.runtime.onMessage.addListener((request, _sender, sendResponse) => {
  if (request.action === 'extractPageInfo') {
    (async () => {
      if (cachedPageInfo) {
        sendResponse(cachedPageInfo);
      } else {
        const pageInfo = await extractGHLPageInfoAsync();
        cachedPageInfo = pageInfo;
        sendResponse(cachedPageInfo);
      }
    })();
    return true;
  }

  if (request.action === 'openPageBuilder') {
    (async () => {
      const pageInfo = cachedPageInfo || await extractGHLPageInfoAsync();
      cachedPageInfo = pageInfo; // Update cache
      const builderUrl = generateBuilderUrl(pageInfo);

      if (builderUrl) {
        window.open(builderUrl, "_blank");
        showToast("Opening page builder...");
        sendResponse({ success: true, url: builderUrl });
      } else {
        showToast("⚠️ Unable to generate builder URL");
        sendResponse({
          success: false,
          error: "Could not generate builder URL",
        });
      }
    })();
    return true;
  }

  if (request.action === 'copyPageId') {
    (async () => {
      const data = cachedPageInfo || await extractGHLPageInfoAsync();
      cachedPageInfo = data; // Update cache
      const textToCopy = data.pageId || data.stepId || "No ID found";
      navigator.clipboard
        .writeText(textToCopy)
        .then(() => {
          showToast(`Copied: ${textToCopy}`);
          sendResponse({ success: true });
        })
        .catch((err) => {
          sendResponse({ success: false, error: err.message });
        });
    })();
    return true;
  }
});
