// GHL Utils - Quick Navigation & Performance Booster
// Keyboard shortcuts, command palette, and recent resources cache

// State management
let quickNavState = {
  enabled: true,
  recentResources: [],
  maxRecents: 10,
  commandPaletteOpen: false,
  keySequence: '',
  keySequenceTimeout: null
};

// Navigation patterns
const NAVIGATION_PATTERNS = {
  contacts: '/contacts',
  workflows: '/workflow',
  opportunities: '/opportunities',
  conversations: '/conversations',
  calendars: '/calendars',
  'funnels-websites': '/funnels-websites',
  dashboard: '/dashboard',
  settings: '/settings',
  automations: '/workflow'
};

// Initialize quick navigation
async function initQuickNavigation() {
  // Load settings
  const stored = await chrome.storage.sync.get(['quickNavEnabled', 'recentResources']);
  quickNavState.enabled = stored.quickNavEnabled !== false; // Enabled by default
  quickNavState.recentResources = stored.recentResources || [];

  if (!quickNavState.enabled) {
    return;
  }

  // Set up keyboard shortcuts
  setupKeyboardShortcuts();

  // Track current page for recents
  trackCurrentPage();
}

// Setup keyboard shortcuts
function setupKeyboardShortcuts() {
  let gKeyPressed = false;
  let gKeyTimeout = null;

  document.addEventListener('keydown', (e) => {
    // Ignore if typing in input/textarea
    if (e.target.matches('input, textarea, [contenteditable]')) {
      return;
    }

    // Cmd+Shift+K / Ctrl+Shift+K - Open command palette
    if ((e.metaKey || e.ctrlKey) && e.shiftKey && e.key === 'K') {
      e.preventDefault();
      openCommandPalette();
      return;
    }

    // Forward slash (/) - Open command palette
    if (e.key === '/') {
      e.preventDefault();
      openCommandPalette();
      return;
    }

    // 'g' key sequences (g+c, g+w, g+o, etc.)
    if (e.key === 'g' && !gKeyPressed) {
      gKeyPressed = true;

      // Reset after 1 second
      clearTimeout(gKeyTimeout);
      gKeyTimeout = setTimeout(() => {
        gKeyPressed = false;
      }, 1000);
      return;
    }

    // Handle second key after 'g'
    if (gKeyPressed) {
      gKeyPressed = false;
      clearTimeout(gKeyTimeout);

      const shortcuts = {
        'c': 'contacts',
        'w': 'workflows',
        'o': 'opportunities',
        'v': 'conversations',
        'a': 'calendars',
        'f': 'funnels-websites',
        'd': 'dashboard',
        's': 'settings'
      };

      const destination = shortcuts[e.key];
      if (destination) {
        e.preventDefault();
        navigateToSection(destination);
        showToast(`Navigating to ${destination}...`);
      }
    }

    // Escape key - Close command palette
    if (e.key === 'Escape' && quickNavState.commandPaletteOpen) {
      e.preventDefault();
      closeCommandPalette();
    }
  });
}

// Open command palette
function openCommandPalette() {
  if (quickNavState.commandPaletteOpen) {
    return;
  }

  quickNavState.commandPaletteOpen = true;

  // Create overlay
  const overlay = document.createElement('div');
  overlay.id = 'ghl-utils-quick-nav-overlay';
  overlay.style.cssText = `
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    backdrop-filter: blur(4px);
    z-index: 999999999;
    display: flex;
    align-items: flex-start;
    justify-content: center;
    padding-top: 100px;
    animation: fadeIn 0.2s ease-out;
  `;

  overlay.addEventListener('click', (e) => {
    if (e.target === overlay) {
      closeCommandPalette();
    }
  });

  // Create command palette
  const palette = document.createElement('div');
  palette.id = 'ghl-utils-command-palette';
  palette.style.cssText = `
    background: white;
    border-radius: 12px;
    width: 90%;
    max-width: 600px;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideInDown 0.3s ease-out;
    overflow: hidden;
  `;

  palette.innerHTML = `
    <div style="
      padding: 20px;
      background: linear-gradient(135deg, #FB8500 0%, #FF9500 100%);
      color: white;
      display: flex;
      justify-content: space-between;
      align-items: center;
      border-radius: 12px 12px 0 0;
    ">
      <h2 style="margin: 0; font-size: 20px; font-weight: 600;">GHL Utils</h2>
      <button id="ghl-utils-close-palette" style="
        background: rgba(255, 255, 255, 0.2);
        border: none;
        color: white;
        font-size: 28px;
        cursor: pointer;
        width: 36px;
        height: 36px;
        border-radius: 6px;
        display: flex;
        align-items: center;
        justify-content: center;
        transition: background 0.2s;
        line-height: 1;
      ">&times;</button>
    </div>

    <div style="padding: 20px; background: #fef9f5;">
      <input
        type="text"
        id="ghl-utils-quick-nav-input"
        placeholder="Type /workflow, /page, /form, /survey or paste ID..."
        style="
          width: 100%;
          padding: 12px 16px;
          border: 2px solid #FB8500;
          border-radius: 8px;
          outline: none;
          font-size: 14px;
          font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
          box-sizing: border-box;
        "
        autofocus
      />
    </div>

    <div id="ghl-utils-search-results-container" style="
      display: none;
      padding: 0 20px;
      background: #fef9f5;
      max-height: 400px;
      overflow-y: auto;
    "></div>

    <div style="padding: 0 20px 20px; background: #fef9f5;">
      <div style="
        background: white;
        border-radius: 8px;
        padding: 16px;
        box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
      ">
        <div style="font-size: 13px; line-height: 1.8; color: #374151;">
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/workflow</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/w</code>
            <span style="margin-left: 8px; color: #6b7280;">- Open workflow</span>
          </div>
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/page</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/p</code>
            <span style="margin-left: 8px; color: #6b7280;">- Open page/funnel</span>
          </div>
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/form</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/f</code>
            <span style="margin-left: 8px; color: #6b7280;">- Open form</span>
          </div>
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/survey</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/s</code>
            <span style="margin-left: 8px; color: #6b7280;">- Open survey</span>
          </div>
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/contact</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/c</code>
            <span style="margin-left: 8px; color: #6b7280;">- Search contacts and view details</span>
          </div>
          <div style="margin-bottom: 8px;">
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/audit</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/a</code>
            <span style="margin-left: 8px; color: #6b7280;">- Search contacts and view audit logs</span>
          </div>
          <div>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/builder</code>
            <span style="margin: 0 4px; color: #9ca3af;">or</span>
            <code style="
              background: #fef3c7;
              padding: 2px 6px;
              border-radius: 3px;
              font-size: 12px;
              color: #92400e;
            ">/b</code>
            <span style="margin-left: 8px; color: #6b7280;">- Open current page in builder</span>
          </div>
        </div>
      </div>
    </div>
  `;

  overlay.appendChild(palette);
  document.body.appendChild(overlay);

  // Close button handler
  document.getElementById('ghl-utils-close-palette').addEventListener('click', closeCommandPalette);
  document.getElementById('ghl-utils-close-palette').addEventListener('mouseenter', (e) => {
    e.target.style.background = 'rgba(255, 255, 255, 0.3)';
  });
  document.getElementById('ghl-utils-close-palette').addEventListener('mouseleave', (e) => {
    e.target.style.background = 'rgba(255, 255, 255, 0.2)';
  });

  // Set up input
  const input = document.getElementById('ghl-utils-quick-nav-input');
  let searchTimeout = null;

  // Handle input changes for search
  input.addEventListener('input', (e) => {
    const value = e.target.value.trim();

    // Clear previous timeout
    clearTimeout(searchTimeout);

    // Check if it's a workflow search (starts with /w or /workflow followed by space and text)
    const workflowSearchMatch = value.match(/^\/(w|workflow)\s+(.+)$/i);

    // Check if it's a contact/audit search (starts with /a or /audit or /c or /contact followed by space and text)
    const contactSearchMatch = value.match(/^\/(a|audit|c|contact)\s+(.+)$/i);

    if (workflowSearchMatch && workflowSearchMatch[2]) {
      const searchQuery = workflowSearchMatch[2];
      // Debounce search
      searchTimeout = setTimeout(() => {
        searchWorkflows(searchQuery);
      }, 300);
    } else if (contactSearchMatch && contactSearchMatch[2]) {
      const commandType = contactSearchMatch[1].toLowerCase();
      const searchQuery = contactSearchMatch[2];
      // Determine if this is for audit logs or contact details
      const isAudit = commandType === 'a' || commandType === 'audit';
      // Debounce search
      searchTimeout = setTimeout(() => {
        searchContacts(searchQuery, isAudit);
      }, 300);
    } else {
      // Hide results if not a search query
      hideSearchResults();
    }
  });

  // Handle Enter key for slash commands
  input.addEventListener('keydown', (e) => {
    if (e.key === 'Enter') {
      const value = input.value.trim().toLowerCase();

      // Check if it's a workflow search with results
      const resultsContainer = document.getElementById('ghl-utils-search-results-container');
      if (resultsContainer && resultsContainer.style.display !== 'none') {
        // If there are results, click the first one (workflow or contact)
        const firstResult = resultsContainer.querySelector('[data-workflow-id], [data-contact-id]');
        if (firstResult) {
          firstResult.click();
          return;
        }
      }

      // Check if it's a contact ID after /a, /audit, /c, or /contact (format: /a {contactId} or /c {contactId})
      const contactIdMatch = input.value.trim().match(/^\/(a|audit|c|contact)\s+([A-Za-z0-9_-]{18,28})$/);
      if (contactIdMatch && contactIdMatch[2]) {
        e.preventDefault();
        const commandType = contactIdMatch[1].toLowerCase();
        const isAudit = commandType === 'a' || commandType === 'audit';
        if (isAudit) {
          openAuditLog(contactIdMatch[2]);
        } else {
          openContact(contactIdMatch[2]);
        }
        closeCommandPalette();
        return;
      }

      // Check if it's a slash command
      if (value.startsWith('/')) {
        // Special handling for /audit, /a, /contact, and /c - show contact search interface
        if (value === '/audit' || value === '/a' || value === '/contact' || value === '/c') {
          e.preventDefault();
          // Update placeholder to prompt for contact search
          const placeholderText = (value === '/a' || value === '/audit')
            ? 'Search for a contact by name or enter contact ID for audit logs...'
            : 'Search for a contact by name or enter contact ID...';
          input.placeholder = placeholderText;
          input.value = value + ' ';
          return;
        }

        const slashCommands = {
          '/workflow': 'workflows',
          '/w': 'workflows',
          '/page': 'funnels-websites',
          '/p': 'funnels-websites',
          '/form': 'funnels-websites',
          '/f': 'funnels-websites',
          '/survey': 'funnels-websites',
          '/s': 'funnels-websites',
          '/contacts': 'contacts', // Keep /contacts for going to contacts list
          '/opportunities': 'opportunities',
          '/o': 'opportunities',
          '/conversations': 'conversations',
          '/v': 'conversations',
          '/calendars': 'calendars',
          '/calendar': 'calendars',
          '/dashboard': 'dashboard',
          '/d': 'dashboard',
          '/settings': 'settings'
        };

        const destination = slashCommands[value];
        if (destination) {
          e.preventDefault();
          navigateToSection(destination);
          closeCommandPalette();
          return;
        }
      }
    }

    // Handle Escape to close
    if (e.key === 'Escape') {
      e.preventDefault();
      closeCommandPalette();
    }
  });

  // Focus input
  input.focus();
}

// Search workflows
async function searchWorkflows(query) {
  const resultsContainer = document.getElementById('ghl-utils-search-results-container');
  if (!resultsContainer) return;

  // Show loading state
  resultsContainer.style.display = 'block';
  resultsContainer.innerHTML = `
    <div style="
      background: white;
      border-radius: 8px;
      padding: 16px;
      margin-bottom: 20px;
      text-align: center;
      color: #6b7280;
    ">
      <div>Searching workflows...</div>
    </div>
  `;

  try {
    // Check if API is available
    if (!window.ghlUtilsAPI) {
      throw new Error('API not available');
    }

    // Wait for API to be ready
    await window.ghlUtilsAPI.waitForReady();

    // Get location ID
    const locationId = window.ghlUtilsAPI.getLocationId();
    if (!locationId) {
      throw new Error('Location ID not found');
    }

    // Search workflows
    const endpoint = `/workflow/${locationId}/list?search=${encodeURIComponent(query)}&limit=50&offset=0&sortBy=name&sortOrder=asc&includeCustomObjects=true&includeObjectiveBuilder=true`;
    const response = await window.ghlUtilsAPI.get(endpoint);

    if (response.status !== 200) {
      throw new Error(`API error: ${response.status}`);
    }

    // API returns {count, rows, isLocationRateLimited}
    const workflows = response.data.rows || [];
    console.log('[Quick Nav] Workflow search results:', workflows.length, response.data);

    // Display results
    if (workflows.length === 0) {
      resultsContainer.innerHTML = `
        <div style="
          background: white;
          border-radius: 8px;
          padding: 16px;
          margin-bottom: 20px;
          text-align: center;
          color: #6b7280;
        ">
          <div>No workflows found for "${query}"</div>
        </div>
      `;
    } else {
      resultsContainer.innerHTML = `
        <div style="
          background: white;
          border-radius: 8px;
          padding: 8px;
          margin-bottom: 20px;
          box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        ">
          <div style="
            padding: 8px 12px;
            font-size: 12px;
            font-weight: 600;
            color: #6b7280;
            text-transform: uppercase;
            letter-spacing: 0.05em;
          ">
            ${workflows.length} workflow${workflows.length !== 1 ? 's' : ''} found
          </div>
          ${workflows.map(workflow => `
            <div
              data-workflow-id="${workflow._id}"
              style="
                padding: 12px;
                border-radius: 6px;
                cursor: pointer;
                transition: background 0.2s;
                border-bottom: 1px solid #f3f4f6;
              "
              onmouseover="this.style.background='#fef9f5'"
              onmouseout="this.style.background='transparent'"
            >
              <div style="
                font-size: 14px;
                font-weight: 500;
                color: #111827;
                margin-bottom: 4px;
              ">${workflow.name || 'Unnamed Workflow'}</div>
              <div style="
                font-size: 12px;
                color: #6b7280;
                font-family: Monaco, Menlo, monospace;
              ">${workflow._id}</div>
            </div>
          `).join('')}
        </div>
      `;

      // Add click handlers to results
      workflows.forEach(workflow => {
        const element = resultsContainer.querySelector(`[data-workflow-id="${workflow._id}"]`);
        if (element) {
          // Left click handler
          element.addEventListener('click', (e) => {
            e.preventDefault();
            const shouldOpenInNewTab = e.ctrlKey || e.metaKey;
            openWorkflow(workflow._id, shouldOpenInNewTab);
            if (!shouldOpenInNewTab) {
              closeCommandPalette();
            }
          });

          // Middle click handler (mouse wheel click)
          element.addEventListener('auxclick', (e) => {
            if (e.button === 1) { // Middle click
              e.preventDefault();
              openWorkflow(workflow._id, true);
            }
          });

          // Context menu handler
          element.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            showContextMenu(e, {
              type: 'workflow',
              id: workflow._id,
              name: workflow.name
            });
          });
        }
      });
    }
  } catch (error) {
    console.error('[Quick Nav] Workflow search error:', error);
    resultsContainer.innerHTML = `
      <div style="
        background: white;
        border-radius: 8px;
        padding: 16px;
        margin-bottom: 20px;
        text-align: center;
        color: #ef4444;
      ">
        <div>Error searching workflows: ${error.message}</div>
      </div>
    `;
  }
}

// Search contacts
async function searchContacts(query, isAudit = false) {
  const resultsContainer = document.getElementById('ghl-utils-search-results-container');
  if (!resultsContainer) return;

  // Show loading state
  resultsContainer.style.display = 'block';
  resultsContainer.innerHTML = `
    <div style="
      background: white;
      border-radius: 8px;
      padding: 16px;
      margin-bottom: 20px;
      text-align: center;
      color: #6b7280;
    ">
      <div>Searching contacts...</div>
    </div>
  `;

  try {
    // Check if API is available
    if (!window.ghlUtilsAPI) {
      throw new Error('API not available');
    }

    // Wait for API to be ready
    await window.ghlUtilsAPI.waitForReady();

    // Get location ID
    const locationId = window.ghlUtilsAPI.getLocationId();
    if (!locationId) {
      throw new Error('Location ID not found');
    }

    // Build the search payload matching GHL's contact search format
    const payload = {
      locationId: locationId,
      page: 1,
      pageLimit: 20,
      sort: [],
      filters: [{
        group: "OR",
        filters: [{
          group: "AND",
          filters: [{
            id: crypto.randomUUID(),
            filterName: "generic",
            filterName_lc: "generic",
            extras: {},
            selectedOption: {
              filterName: "generic",
              filterName_lc: "generic",
              condition: "is",
              firstValue: query,
              secondValue: ""
            }
          }]
        }]
      }]
    };

    // Make the search request
    const response = await window.ghlUtilsAPI.post('/contacts/search/2', payload);

    if (response.status !== 200) {
      throw new Error(`API error: ${response.status}`);
    }

    const contacts = response.data.contacts || [];
    console.log('[Quick Nav] Contact search results:', contacts.length, response.data);

    // Display results
    if (contacts.length === 0) {
      resultsContainer.innerHTML = `
        <div style="
          background: white;
          border-radius: 8px;
          padding: 16px;
          margin-bottom: 20px;
          text-align: center;
          color: #6b7280;
        ">
          <div>No contacts found for "${query}"</div>
          <div style="margin-top: 12px; font-size: 12px;">
            <strong>Tip:</strong> Enter a contact ID directly to view audit logs
          </div>
        </div>
      `;
    } else {
      resultsContainer.innerHTML = `
        <div style="
          background: white;
          border-radius: 8px;
          padding: 8px;
          margin-bottom: 20px;
          box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        ">
          <div style="
            padding: 8px 12px;
            font-size: 12px;
            font-weight: 600;
            color: #6b7280;
            text-transform: uppercase;
            letter-spacing: 0.05em;
          ">
            ${contacts.length} contact${contacts.length !== 1 ? 's' : ''} found
          </div>
          ${contacts.map(contact => `
            <div
              data-contact-id="${contact.id}"
              style="
                padding: 12px;
                border-radius: 6px;
                cursor: pointer;
                transition: background 0.2s;
                border-bottom: 1px solid #f3f4f6;
              "
              onmouseover="this.style.background='#fef9f5'"
              onmouseout="this.style.background='transparent'"
            >
              <div style="
                font-size: 14px;
                font-weight: 500;
                color: #111827;
                margin-bottom: 4px;
              ">${contact.contactName || contact.firstName + ' ' + contact.lastName || 'Unnamed Contact'}</div>
              <div style="
                font-size: 12px;
                color: #6b7280;
                margin-bottom: 2px;
              ">${contact.email || ''} ${contact.phone ? '• ' + contact.phone : ''}</div>
              <div style="
                font-size: 11px;
                color: #9ca3af;
                font-family: Monaco, Menlo, monospace;
              ">${contact.id}</div>
            </div>
          `).join('')}
        </div>
      `;

      // Add click handlers to results
      contacts.forEach(contact => {
        const element = resultsContainer.querySelector(`[data-contact-id="${contact.id}"]`);
        if (element) {
          // Left click handler
          element.addEventListener('click', (e) => {
            e.preventDefault();
            const shouldOpenInNewTab = e.ctrlKey || e.metaKey;
            if (isAudit) {
              openAuditLog(contact.id, shouldOpenInNewTab);
            } else {
              openContact(contact.id, shouldOpenInNewTab);
            }
            if (!shouldOpenInNewTab) {
              closeCommandPalette();
            }
          });

          // Middle click handler (mouse wheel click)
          element.addEventListener('auxclick', (e) => {
            if (e.button === 1) { // Middle click
              e.preventDefault();
              if (isAudit) {
                openAuditLog(contact.id, true);
              } else {
                openContact(contact.id, true);
              }
            }
          });

          // Context menu handler
          element.addEventListener('contextmenu', (e) => {
            e.preventDefault();
            showContextMenu(e, {
              type: isAudit ? 'audit' : 'contact',
              id: contact.id,
              name: contact.contactName || contact.firstName + ' ' + contact.lastName || 'Contact'
            });
          });
        }
      });
    }
  } catch (error) {
    console.error('[Quick Nav] Contact search error:', error);
    resultsContainer.innerHTML = `
      <div style="
        background: white;
        border-radius: 8px;
        padding: 16px;
        margin-bottom: 20px;
        text-align: center;
        color: #ef4444;
      ">
        <div>Error searching contacts: ${error.message}</div>
      </div>
    `;
  }
}

// Hide search results
function hideSearchResults() {
  const resultsContainer = document.getElementById('ghl-utils-search-results-container');
  if (resultsContainer) {
    resultsContainer.style.display = 'none';
    resultsContainer.innerHTML = '';
  }
}

// Open workflow by ID
function openWorkflow(workflowId, openInNewTab = false) {
  const locationId = getCurrentLocationId();
  if (!locationId) {
    showToast('⚠️ Could not detect location ID');
    return;
  }

  const url = `https://app.gohighlevel.com/location/${locationId}/workflow/${workflowId}`;
  if (openInNewTab) {
    window.open(url, '_blank');
  } else {
    window.location.href = url;
  }
}

// Open contact detail page
function openContact(contactId, openInNewTab = false) {
  const locationId = getCurrentLocationId();
  if (!locationId) {
    showToast('⚠️ Could not detect location ID');
    return;
  }

  const url = `https://app.gohighlevel.com/v2/location/${locationId}/contacts/detail/${contactId}`;
  if (openInNewTab) {
    window.open(url, '_blank');
  } else {
    window.location.href = url;
  }
}

// Open audit log for contact
function openAuditLog(contactId, openInNewTab = false) {
  const locationId = getCurrentLocationId();
  if (!locationId) {
    showToast('⚠️ Could not detect location ID');
    return;
  }

  const url = `https://app.gohighlevel.com/v2/location/${locationId}/settings/audit/logs?documentId=${contactId}`;
  if (openInNewTab) {
    window.open(url, '_blank');
  } else {
    window.location.href = url;
  }
}

// Show context menu for search results
function showContextMenu(event, item) {
  // Remove existing context menu if any
  const existingMenu = document.getElementById('ghl-utils-context-menu');
  if (existingMenu) {
    existingMenu.remove();
  }

  // Create context menu
  const menu = document.createElement('div');
  menu.id = 'ghl-utils-context-menu';
  menu.style.cssText = `
    position: fixed;
    left: ${event.clientX}px;
    top: ${event.clientY}px;
    background: white;
    border: 1px solid #e5e7eb;
    border-radius: 8px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.2);
    z-index: 1000000000;
    padding: 4px;
    min-width: 200px;
    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
  `;

  const menuItems = [
    {
      label: 'Open',
      icon: '↗',
      action: () => {
        if (item.type === 'workflow') {
          openWorkflow(item.id, false);
          closeCommandPalette();
        } else if (item.type === 'contact') {
          openContact(item.id, false);
          closeCommandPalette();
        } else if (item.type === 'audit') {
          openAuditLog(item.id, false);
          closeCommandPalette();
        }
      }
    },
    {
      label: 'Open in New Tab',
      icon: '⧉',
      action: () => {
        if (item.type === 'workflow') {
          openWorkflow(item.id, true);
        } else if (item.type === 'contact') {
          openContact(item.id, true);
        } else if (item.type === 'audit') {
          openAuditLog(item.id, true);
        }
      }
    },
    {
      label: 'Copy ID',
      icon: '📋',
      action: () => {
        navigator.clipboard.writeText(item.id).then(() => {
          showToast('✓ ID copied to clipboard');
        });
      }
    }
  ];

  menu.innerHTML = menuItems.map(menuItem => `
    <div class="ghl-utils-context-menu-item" style="
      padding: 8px 12px;
      cursor: pointer;
      border-radius: 4px;
      font-size: 13px;
      color: #374151;
      display: flex;
      align-items: center;
      gap: 8px;
      transition: background 0.15s;
    ">
      <span style="font-size: 14px; width: 16px;">${menuItem.icon}</span>
      <span>${menuItem.label}</span>
    </div>
  `).join('');

  document.body.appendChild(menu);

  // Add click handlers
  const menuItemElements = menu.querySelectorAll('.ghl-utils-context-menu-item');
  menuItemElements.forEach((element, index) => {
    element.addEventListener('click', () => {
      menuItems[index].action();
      menu.remove();
    });

    element.addEventListener('mouseenter', () => {
      element.style.background = '#fef9f5';
    });

    element.addEventListener('mouseleave', () => {
      element.style.background = 'transparent';
    });
  });

  // Close menu when clicking outside
  const closeMenu = (e) => {
    if (!menu.contains(e.target)) {
      menu.remove();
      document.removeEventListener('click', closeMenu);
    }
  };

  // Delay to prevent immediate close from the same click
  setTimeout(() => {
    document.addEventListener('click', closeMenu);
  }, 10);

  // Adjust position if menu goes off screen
  setTimeout(() => {
    const rect = menu.getBoundingClientRect();
    if (rect.right > window.innerWidth) {
      menu.style.left = `${event.clientX - rect.width}px`;
    }
    if (rect.bottom > window.innerHeight) {
      menu.style.top = `${event.clientY - rect.height}px`;
    }
  }, 0);
}

// Close command palette
function closeCommandPalette() {
  const overlay = document.getElementById('ghl-utils-quick-nav-overlay');
  if (overlay) {
    overlay.remove();
  }
  quickNavState.commandPaletteOpen = false;

  // Also remove any context menus
  const contextMenu = document.getElementById('ghl-utils-context-menu');
  if (contextMenu) {
    contextMenu.remove();
  }
}


// Navigate to section
function navigateToSection(section) {
  const locationId = getCurrentLocationId();
  if (!locationId) {
    showToast('⚠️ Could not detect location ID');
    return;
  }

  const path = NAVIGATION_PATTERNS[section];
  if (path) {
    const url = `https://app.gohighlevel.com/v2/location/${locationId}${path}`;
    window.location.href = url;
  }
}

// Track current page for recents
function trackCurrentPage() {
  const url = window.location.href;
  const title = document.title;

  // Detect page type
  let type = 'page';
  if (url.includes('/contacts')) type = 'contacts';
  else if (url.includes('/workflow')) type = 'workflow';
  else if (url.includes('/opportunities')) type = 'opportunities';
  else if (url.includes('/conversations')) type = 'conversations';
  else if (url.includes('/calendars')) type = 'calendars';
  else if (url.includes('/funnels-websites')) type = 'funnels & websites';

  // Add to recents
  addToRecents({
    url,
    title,
    type,
    timestamp: Date.now()
  });
}

// Add to recent resources
function addToRecents(resource) {
  // Remove duplicates
  quickNavState.recentResources = quickNavState.recentResources.filter(
    r => r.url !== resource.url
  );

  // Add to beginning
  quickNavState.recentResources.unshift(resource);

  // Limit size
  if (quickNavState.recentResources.length > quickNavState.maxRecents) {
    quickNavState.recentResources = quickNavState.recentResources.slice(0, quickNavState.maxRecents);
  }

  // Save to storage
  chrome.storage.sync.set({ recentResources: quickNavState.recentResources });
}

// Get current location ID
function getCurrentLocationId() {
  const match = window.location.href.match(/\/location\/([A-Za-z0-9_-]{18,28})/);
  return match ? match[1] : null;
}

// Get icon for resource type
function getIconForType(type) {
  const icons = {
    'contacts': '👤',
    'workflow': '⚙️',
    'opportunities': '💼',
    'conversations': '💬',
    'calendars': '📅',
    'funnels & websites': '📄',
    'page': '📄'
  };
  return icons[type] || '📄';
}

// Get relative time
function getRelativeTime(timestamp) {
  const seconds = Math.floor((Date.now() - timestamp) / 1000);

  if (seconds < 60) return 'Just now';
  if (seconds < 3600) return `${Math.floor(seconds / 60)}m ago`;
  if (seconds < 86400) return `${Math.floor(seconds / 3600)}h ago`;
  return `${Math.floor(seconds / 86400)}d ago`;
}

// Show toast (use existing function or create inline)
function showToast(message) {
  if (typeof window.ghlUtilsShowToast === 'function') {
    window.ghlUtilsShowToast(message);
  }
}

// Initialize when content script loads
if (document.readyState === 'loading') {
  document.addEventListener('DOMContentLoaded', initQuickNavigation);
} else {
  initQuickNavigation();
}

// Re-track page on URL changes (SPA navigation)
let lastQuickNavUrl = location.href;
new MutationObserver(() => {
  if (location.href !== lastQuickNavUrl) {
    lastQuickNavUrl = location.href;
    if (quickNavState.enabled) {
      trackCurrentPage();
    }
  }
}).observe(document.querySelector('head > title') || document.body, {
  subtree: true,
  characterData: true,
  childList: true
});

// Listen for settings updates from popup
chrome.runtime.onMessage.addListener((request, _sender, sendResponse) => {
  if (request.action === 'updateQuickNavSetting') {
    quickNavState.enabled = request.enabled;

    // Close command palette if open and quick nav is disabled
    if (!quickNavState.enabled && quickNavState.commandPaletteOpen) {
      closeCommandPalette();
    }

    sendResponse({ success: true });
  }
});
