// GHL URL patterns - Update these with your actual GHL account subdomain
const GHL_PATTERNS = {
  workflow: 'https://app.gohighlevel.com/location/{locationId}/workflow/{id}',
  page: 'https://app.gohighlevel.com/v2/location/{locationId}/funnels-websites/funnels/{id}',
  form: 'https://app.gohighlevel.com/v2/location/{locationId}/form-builder-v2/{id}',
  survey: 'https://app.gohighlevel.com/v2/location/{locationId}/survey-builder-v2/{id}',
  contact: 'https://app.gohighlevel.com/v2/location/{locationId}/contacts/detail/{id}',
  calendar: 'https://app.gohighlevel.com/v2/location/{locationId}/calendars/{id}', // Placeholder - needs actual URL
  integration: 'https://app.gohighlevel.com/v2/location/{locationId}/settings/integrations/{id}', // Placeholder - needs actual URL
  audit: 'https://app.gohighlevel.com/v2/location/{locationId}/settings/audit/logs?documentId={id}'
};

// Command definitions with aliases
const COMMANDS = [
  {
    command: '/workflow',
    aliases: ['/w'],
    icon: '⚙️',
    description: 'Open workflow by ID',
    type: 'workflow'
  },
  {
    command: '/page',
    aliases: ['/p'],
    icon: '📄',
    description: 'Open funnel/page by ID',
    type: 'page'
  },
  {
    command: '/form',
    aliases: ['/f'],
    icon: '📝',
    description: 'Open form by ID',
    type: 'form'
  },
  {
    command: '/survey',
    aliases: ['/s'],
    icon: '📊',
    description: 'Open survey by ID',
    type: 'survey'
  },
  {
    command: '/contact',
    aliases: ['/c'],
    icon: '👤',
    description: 'Open contact by ID',
    type: 'contact'
  },
  {
    command: '/audit',
    aliases: ['/a'],
    icon: '📋',
    description: 'Open audit logs by ID',
    type: 'audit'
  }
];

const searchInput = document.getElementById('searchInput');
const goButton = document.getElementById('goButton');
const suggestionsDiv = document.getElementById('suggestions');

// Load saved location ID
let locationId = '';
chrome.storage.sync.get(['locationId'], (result) => {
  if (result.locationId) {
    locationId = result.locationId;
  } else {
    // Try to auto-detect from current tab before showing manual prompt
    attemptAutoDetection();
  }
});

function attemptAutoDetection() {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0] && tabs[0].url) {
      const url = tabs[0].url;
      const locationMatch = url.match(/\/location\/([A-Za-z0-9_-]{18,28})/);

      if (locationMatch && locationMatch[1]) {
        // Found location ID in current tab URL
        const detectedLocationId = locationMatch[1];
        chrome.storage.sync.set({ locationId: detectedLocationId }, () => {
          locationId = detectedLocationId;
        });
      } else {
        // Could not auto-detect, show manual prompt
        showLocationPrompt();
      }
    } else {
      // No active tab or URL, show manual prompt
      showLocationPrompt();
    }
  });
}

function showLocationPrompt() {
  const errorDiv = document.createElement('div');
  errorDiv.className = 'error-message';
  errorDiv.innerHTML = `
    <strong>Location ID not detected</strong><br>
    Please navigate to any GHL page to auto-detect your location ID.
  `;
  document.querySelector('.container').insertBefore(errorDiv, document.querySelector('.search-box'));
}

// Input event listener for suggestions
searchInput.addEventListener('input', (e) => {
  const value = e.target.value.trim();

  if (value.startsWith('/')) {
    showSuggestions(value);
  } else {
    suggestionsDiv.innerHTML = '';
  }
});

// Enter key handler
searchInput.addEventListener('keydown', (e) => {
  if (e.key === 'Enter') {
    handleNavigation();
  }
});

// Go button handler
goButton.addEventListener('click', handleNavigation);

function showSuggestions(input) {
  const query = input.toLowerCase();
  const queryCmd = query.split(' ')[0];
  const matches = COMMANDS.filter(cmd =>
    cmd.command.startsWith(queryCmd) ||
    (cmd.aliases && cmd.aliases.some(alias => alias.startsWith(queryCmd)))
  );

  if (matches.length === 0) {
    suggestionsDiv.innerHTML = '';
    return;
  }

  suggestionsDiv.innerHTML = matches.map(cmd => `
    <div class="suggestion-item" data-command="${cmd.command}">
      <span class="suggestion-icon">${cmd.icon}</span>
      <div class="suggestion-text">
        <div class="suggestion-command">${cmd.command}${cmd.aliases ? ` (${cmd.aliases.join(', ')})` : ''}</div>
        <div class="suggestion-desc">${cmd.description}</div>
      </div>
    </div>
  `).join('');

  // Add click handlers to suggestions
  document.querySelectorAll('.suggestion-item').forEach(item => {
    item.addEventListener('click', () => {
      searchInput.value = item.dataset.command + ' ';
      searchInput.focus();
      suggestionsDiv.innerHTML = '';
    });
  });
}

function parseInput(input) {
  const trimmed = input.trim();

  // Check if it starts with a slash command
  if (trimmed.startsWith('/')) {
    const parts = trimmed.split(/\s+/);
    const command = parts[0];
    const id = parts.slice(1).join('').trim();

    // Check for exact command or alias match
    const cmdDef = COMMANDS.find(c =>
      c.command === command ||
      (c.aliases && c.aliases.includes(command))
    );

    if (cmdDef) {
      // Special case: /a or /audit without ID
      if (cmdDef.type === 'audit' && !id) {
        // Try to get contact ID from active tab URL via content script
        chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
          if (tabs[0]) {
            chrome.tabs.sendMessage(tabs[0].id, { action: 'getContactId' }, (response) => {
              if (response && response.contactId) {
                handleNavigation('audit', response.contactId);
                window.close();
              } else {
                showError('No contact ID found. Please specify an ID or open this on a contact page.');
              }
            });
          }
        });
        return 'pending'; // Special flag to prevent immediate navigation
      }

      // Normal case: command with ID
      if (id) {
        return { type: cmdDef.type, id };
      }
    }
  }

  // If it's just an ID, try to detect the type or default to workflow
  if (trimmed.length > 0) {
    return { type: 'workflow', id: trimmed };
  }

  return null;
}

function buildUrl(type, id) {
  if (!locationId) {
    showLocationPrompt();
    return null;
  }

  const pattern = GHL_PATTERNS[type];
  if (!pattern) return null;

  return pattern.replace('{locationId}', locationId).replace('{id}', id);
}

function handleNavigation() {
  const input = searchInput.value;
  const parsed = parseInput(input);

  // Check if it's pending (async operation in progress)
  if (parsed === 'pending') {
    return; // parseInput will handle it asynchronously
  }

  if (!parsed) {
    showError('Please enter a valid command or ID');
    return;
  }

  const url = buildUrl(parsed.type, parsed.id);
  if (!url) {
    showError('Invalid resource type');
    return;
  }

  // Open in new tab
  chrome.tabs.create({ url });

  // Clear input and close popup
  searchInput.value = '';
  window.close();
}

function showError(message) {
  // Remove any existing error
  const existingError = document.querySelector('.error-message');
  if (existingError) existingError.remove();

  const errorDiv = document.createElement('div');
  errorDiv.className = 'error-message';
  errorDiv.textContent = message;
  document.querySelector('.container').insertBefore(errorDiv, document.querySelector('.search-box'));

  setTimeout(() => errorDiv.remove(), 3000);
}

// Focus input when popup opens
searchInput.focus();

// ============================================================================
// QUICK ACTIONS - Page Builder
// ============================================================================

const quickActionsDiv = document.getElementById('quickActions');
const openPageBuilderBtn = document.getElementById('openPageBuilder');
const workflowActionsDiv = document.getElementById('workflowActions');
const analyzeWorkflowBtn = document.getElementById('analyzeWorkflow');

// Check if current tab is a GHL page and show quick actions
chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
  if (tabs[0]) {
    const currentTab = tabs[0];

    // Check if it's a GHL page
    const isGHLDomain = currentTab.url && (
      currentTab.url.includes('gohighlevel.com') ||
      currentTab.url.startsWith('https://')
    );

    if (isGHLDomain) {
      // Try to get page info from content script
      chrome.tabs.sendMessage(currentTab.id, { action: 'extractPageInfo' }, (response) => {
        if (chrome.runtime.lastError) {
          // Content script not loaded yet or not a valid page
          return;
        }

        if (response && response.isGHLPage) {
          // Show quick actions section
          quickActionsDiv.style.display = 'block';

          // Update button text if we have page info
          if (response.pageId) {
            const pageType = response.type === 'funnel' ? 'Funnel Step' :
                           response.type === 'website' ? 'Website Page' :
                           response.type === 'page-builder' ? 'Page' : 'Page';
            openPageBuilderBtn.querySelector('.action-text').textContent =
              `Open ${pageType} in Builder`;
          }
        }
      });

      // Check if on a workflow page
      chrome.tabs.sendMessage(tabs[0].id, { action: 'getWorkflowId' }, (response) => {
        if (chrome.runtime.lastError) {
          // Silently ignore if content script not ready
          return;
        }

        if (response && response.workflowId) {
          // Show workflow actions section
          workflowActionsDiv.style.display = 'block';
          // Store workflow ID for button click
          analyzeWorkflowBtn.dataset.workflowId = response.workflowId;
        }
      });
    }
  }
});

// Handle "Open in Page Builder" button click
openPageBuilderBtn.addEventListener('click', () => {
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]) {
      chrome.tabs.sendMessage(tabs[0].id, { action: 'openPageBuilder' }, (response) => {
        if (chrome.runtime.lastError) {
          showError('Unable to open page builder. Are you on a GHL page?');
          return;
        }

        if (response && response.success) {
          window.close();
        } else {
          showError(response?.error || 'Unable to open page builder');
        }
      });
    }
  });
});

// Handle "Analyze Workflow" button click
analyzeWorkflowBtn.addEventListener('click', () => {
  const workflowId = analyzeWorkflowBtn.dataset.workflowId;
  chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
    if (tabs[0]) {
      chrome.tabs.sendMessage(tabs[0].id, {
        action: 'openWorkflowAnalysis',
        workflowId: workflowId
      }, (response) => {
        if (chrome.runtime.lastError) {
          showError('Unable to analyze workflow. Please refresh the page and try again.');
          return;
        }

        if (response && response.success) {
          window.close();
        } else {
          showError(response?.error || 'Unable to open workflow analyzer');
        }
      });
    }
  });
});

// ============================================================================
// SETTINGS MANAGEMENT
// ============================================================================

const settingsBtn = document.getElementById('settingsBtn');
const settingsPanel = document.getElementById('settingsPanel');
const closeSettingsBtn = document.getElementById('closeSettings');
const cancelSettingsBtn = document.getElementById('cancelSettings');
const saveSettingsBtn = document.getElementById('saveSettings');
const authMethodSelect = document.getElementById('authMethodSelect');
const pitFields = document.getElementById('pitFields');
const apiKeyInput = document.getElementById('apiKeyInput');
const locationIdInput = document.getElementById('locationIdInput');
const openaiApiKeyInput = document.getElementById('openaiApiKeyInput');
const quickNavToggle = document.getElementById('quickNavToggle');
const fieldInspectorToggle = document.getElementById('fieldInspectorToggle');
const aiAnalysisToggle = document.getElementById('aiAnalysisToggle');
const settingsMessage = document.getElementById('settingsMessage');
const helpText = document.querySelector('.help-text');
const searchBox = document.querySelector('.search-box');
const quickActions = document.getElementById('quickActions');
const exportAllWorkflowsBtn = document.getElementById('exportAllWorkflows');
const exportProgress = document.getElementById('exportProgress');
const exportProgressText = document.getElementById('exportProgressText');
const exportProgressBar = document.getElementById('exportProgressBar');

// Handle auth method selection - show/hide PIT fields
authMethodSelect.addEventListener('change', () => {
  if (authMethodSelect.value === 'pit') {
    pitFields.style.display = 'block';
  } else {
    pitFields.style.display = 'none';
  }
});

// Load saved settings
chrome.storage.sync.get(['authMethod', 'ghlApiKey', 'locationId', 'openaiApiKey', 'quickNavEnabled', 'fieldInspectorEnabled', 'aiAnalysisEnabled'], (result) => {
  if (result.authMethod) {
    authMethodSelect.value = result.authMethod;
    if (result.authMethod === 'pit') {
      pitFields.style.display = 'block';
    }
  } else {
    // Default to revex
    authMethodSelect.value = 'revex';
  }
  if (result.ghlApiKey) {
    apiKeyInput.value = result.ghlApiKey;
  }
  if (result.locationId) {
    locationIdInput.value = result.locationId;
  }
  if (result.openaiApiKey) {
    openaiApiKeyInput.value = result.openaiApiKey;
  }
  if (result.quickNavEnabled !== undefined) {
    quickNavToggle.checked = result.quickNavEnabled;
  }
  if (result.fieldInspectorEnabled !== undefined) {
    fieldInspectorToggle.checked = result.fieldInspectorEnabled;
  }
  if (result.aiAnalysisEnabled !== undefined) {
    aiAnalysisToggle.checked = result.aiAnalysisEnabled;
  } else {
    // Default to enabled
    aiAnalysisToggle.checked = true;
  }
});

// Show settings panel
settingsBtn.addEventListener('click', () => {
  // Hide main content
  searchBox.style.display = 'none';
  if (helpText) helpText.style.display = 'none';
  if (quickActions) quickActions.style.display = 'none';

  // Show settings
  settingsPanel.style.display = 'block';

  // Load current values
  chrome.storage.sync.get(['authMethod', 'ghlApiKey', 'locationId', 'openaiApiKey', 'quickNavEnabled', 'fieldInspectorEnabled', 'aiAnalysisEnabled'], (result) => {
    if (result.authMethod) {
      authMethodSelect.value = result.authMethod;
      if (result.authMethod === 'pit') {
        pitFields.style.display = 'block';
      } else {
        pitFields.style.display = 'none';
      }
    } else {
      authMethodSelect.value = 'revex';
      pitFields.style.display = 'none';
    }
    if (result.ghlApiKey) {
      apiKeyInput.value = result.ghlApiKey;
    }
    if (result.locationId) {
      locationIdInput.value = result.locationId;
    }
    if (result.openaiApiKey) {
      openaiApiKeyInput.value = result.openaiApiKey;
    }
    if (result.quickNavEnabled !== undefined) {
      quickNavToggle.checked = result.quickNavEnabled;
    }
    if (result.fieldInspectorEnabled !== undefined) {
      fieldInspectorToggle.checked = result.fieldInspectorEnabled;
    }
    if (result.aiAnalysisEnabled !== undefined) {
      aiAnalysisToggle.checked = result.aiAnalysisEnabled;
    } else {
      aiAnalysisToggle.checked = true;
    }
  });
});

// Close settings panel
function closeSettings() {
  settingsPanel.style.display = 'none';
  searchBox.style.display = 'flex';
  if (helpText) helpText.style.display = 'block';
  if (quickActions && quickActions.querySelector('.action-button')) {
    quickActions.style.display = 'block';
  }
  settingsMessage.style.display = 'none';
}

closeSettingsBtn.addEventListener('click', closeSettings);
cancelSettingsBtn.addEventListener('click', closeSettings);

// Save settings
saveSettingsBtn.addEventListener('click', () => {
  const authMethod = authMethodSelect.value;
  const apiKey = apiKeyInput.value.trim();
  const locId = locationIdInput.value.trim();
  const openaiKey = openaiApiKeyInput.value.trim();
  const quickNavEnabled = quickNavToggle.checked;
  const fieldInspectorEnabled = fieldInspectorToggle.checked;
  const aiAnalysisEnabled = aiAnalysisToggle.checked;

  // Validation - if using PIT, require API key
  if (authMethod === 'pit' && !apiKey) {
    showSettingsMessage('Private Integration Token is required when using PIT authentication', 'error');
    return;
  }

  // Save to storage
  chrome.storage.sync.set({
    authMethod: authMethod,
    ghlApiKey: apiKey,
    locationId: locId,
    openaiApiKey: openaiKey,
    quickNavEnabled: quickNavEnabled,
    fieldInspectorEnabled: fieldInspectorEnabled,
    aiAnalysisEnabled: aiAnalysisEnabled
  }, () => {
    locationId = locId; // Update local variable
    showSettingsMessage('Settings saved successfully! Please refresh the GHL page for changes to take effect.', 'success');

    // Notify content scripts
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
      if (tabs[0]) {
        // Update quick nav setting
        chrome.tabs.sendMessage(tabs[0].id, {
          action: 'updateQuickNavSetting',
          enabled: quickNavEnabled
        }).catch(() => {
          // Content script may not be loaded yet, that's okay
        });

        // Update field inspector setting
        chrome.tabs.sendMessage(tabs[0].id, {
          action: 'updateFieldInspectorSetting',
          enabled: fieldInspectorEnabled
        }).catch(() => {
          // Content script may not be loaded yet, that's okay
        });
      }
    });

    // Close settings after 1.5 seconds
    setTimeout(() => {
      closeSettings();
    }, 1500);
  });
});

// Show settings message
function showSettingsMessage(message, type) {
  const colors = {
    success: { bg: '#d1fae5', text: '#065f46', border: '#6ee7b7' },
    error: { bg: '#fee2e2', text: '#991b1b', border: '#fca5a5' },
    info: { bg: '#dbeafe', text: '#1e40af', border: '#93c5fd' }
  };

  const color = colors[type] || colors.info;

  settingsMessage.textContent = message;
  settingsMessage.style.display = 'block';
  settingsMessage.style.background = color.bg;
  settingsMessage.style.color = color.text;
  settingsMessage.style.border = `1px solid ${color.border}`;
}

// ============================================================================
// WORKFLOW EXPORT
// ============================================================================

// Handle "Export All Workflows" button click
exportAllWorkflowsBtn.addEventListener('click', async () => {
  // Disable button during export
  exportAllWorkflowsBtn.disabled = true;
  exportAllWorkflowsBtn.style.opacity = '0.6';
  exportAllWorkflowsBtn.style.cursor = 'not-allowed';

  // Show progress indicator
  exportProgress.style.display = 'block';
  exportProgressText.textContent = 'Preparing export...';
  exportProgressBar.style.width = '0%';

  try {
    // Get current tab
    const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!tabs[0]) {
      throw new Error('No active tab found');
    }

    // Request export from content script
    chrome.tabs.sendMessage(tabs[0].id, {
      action: 'exportAllWorkflows'
    }, (response) => {
      if (chrome.runtime.lastError) {
        console.error('Export error:', chrome.runtime.lastError);
        showSettingsMessage('Unable to export workflows. Please make sure you are on a GHL page and try again.', 'error');
        resetExportUI();
        return;
      }

      if (!response || !response.success) {
        showSettingsMessage(response?.error || 'Export failed. Please try again.', 'error');
        resetExportUI();
        return;
      }

      // Success - content script will handle the download
      // Listen for progress updates
    });

    // Listen for progress updates from content script
    chrome.runtime.onMessage.addListener(function progressListener(request) {
      if (request.action === 'exportProgress') {
        exportProgressText.textContent = request.message;
        exportProgressBar.style.width = request.progress + '%';
      } else if (request.action === 'exportComplete') {
        showSettingsMessage('Export completed successfully!', 'success');
        resetExportUI();
        chrome.runtime.onMessage.removeListener(progressListener);
      } else if (request.action === 'exportError') {
        showSettingsMessage('Export failed: ' + request.error, 'error');
        resetExportUI();
        chrome.runtime.onMessage.removeListener(progressListener);
      }
    });

  } catch (error) {
    console.error('Export error:', error);
    showSettingsMessage('Export failed: ' + error.message, 'error');
    resetExportUI();
  }
});

function resetExportUI() {
  exportAllWorkflowsBtn.disabled = false;
  exportAllWorkflowsBtn.style.opacity = '1';
  exportAllWorkflowsBtn.style.cursor = 'pointer';

  setTimeout(() => {
    exportProgress.style.display = 'none';
    exportProgressBar.style.width = '0%';
  }, 2000);
}

